/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы Рутокен с криптопровайдером КриптоПро CSP                *
 * с использованием интерфейса CryptoAPI на языке C                       *
 *------------------------------------------------------------------------*
 * Электронная подпись данных и проверка подписи ключевой парой,          *
 * хранящейся на Рутокен:                                                 *
 *  - проверка наличия криптопровайдера КриптоПро CSP в системе;          *
 *  - инициализация криптопровайдера;                                     *
 *  - хэширование данных по алгоритму ГОСТ Р 34.11-2012-256;              *
 *  - подпись хэшированных данных по алгоритму ГОСТ Р 34.10-2012-256;     *
 *  - проверка подписи;                                                   *
 *  - освобождение памяти, дескрипторов и контекстов.                     *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти Рутокен примером         *
 * Create-GOST34.10-2012-256 с типом ключа KEYPAIR_TYPE, установленным    *
 * как AT_SIGNATURE в Common.h.                                           *
 *************************************************************************/

#include "Common.h"

static BYTE pbData[] = {
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
};

int main(void) {
    LPCWSTR szProvNameW =
        CRYPTOPRO_2012_256_PROV_W; // Имя криптопровайдера заменить на CRYPTOPRO_FKN_2012_256_PROV_W для ФКН
    DWORD dwProvType = CRYPTOPRO_2012_256_PROV_TYPE; // Тип криптопровайдера

    HCRYPTPROV hProv = 0; // Дескриптор криптопровайдера
    HCRYPTKEY hKey = 0;   // Дескриптор ключа
    HCRYPTHASH hHash = 0; // Дескриптор хэша

    BYTE* pbHash = NULL;       // Указатель на буфер для хэша
    DWORD dwHashSize = 0;      // Размер хэша
    BYTE* pbSignature = NULL;  // Указатель на буфер для подписи
    DWORD dwSignatureSize = 0; // Размер подписи

    DWORD i = 0;          // Вспомогательная переменная-счетчик
    BOOL bResult = FALSE; // Результат выполнения примера

    for (;;) {
        /**********************************************************************
         * Шаг 1. Проверка наличия выбранного криптопровайдера в системе       *
         **********************************************************************/

        wprintf(L"Checking whether %s provider exists", szProvNameW);
        bResult = CryptAcquireContextW(&hProv, // Указатель на дескриптор криптопровайдера
                                       NULL,        // Имя ключевого контейнера
                                       szProvNameW, // Имя криптопровайдера
                                       dwProvType,  // Тип криптопровайдера
                                       CRYPT_VERIFYCONTEXT); // Флаг операции, не требующей работы с контейнером

        if (!bResult) {
            if (GetLastError() == NTE_KEYSET_NOT_DEF) {
                wprintf(L" -> FAILED \nProvider has not been installed\n");
            } else {
                wprintf(L" -> FAILED, error number: 0x%0.8x\n", GetLastError());
            }
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Освобождение контекста криптопровайдера                             *
         **********************************************************************/
        CryptReleaseContext(hProv, 0);

        /**********************************************************************
         * Шаг 2. Вычисление хэша данных                                       *
         **********************************************************************/
        wprintf(L"Getting data hash...\n");

        /**********************************************************************
         * Шаг 2.1 Инициализация контекста криптопровайдера                    *
         **********************************************************************/
        wprintf(L" Initializing context");
        bResult = CryptAcquireContextW(&hProv, // Указатель на дескриптор криптопровайдера
                                       CONT_NAME_2012_256_W, // Имя ключевого контейнера
                                       szProvNameW,          // Имя криптопровайдера
                                       dwProvType,           // Тип криптопровайдера
                                       0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 2.2 Создание объекта хэша                                       *
         **********************************************************************/
        wprintf(L" Creating hash object");
        bResult = CryptCreateHash(hProv,                 // Дескриптор криптопровайдера
                                  ALG_GOST3411_2012_256, // Идентификатор алгоритма хэширования
                                  0, 0,
                                  &hHash); // Указатель на дескриптор хэша

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 2.3 Задание параметров функции хэширования                      *
         * Задание параметров функции хэширования не предусмотрено в           *
         * ГОСТ Р 34.11-2012.                                                  *
         **********************************************************************/

        /**********************************************************************
         * Шаг 2.4 Вычисление значения хэша                                    *
         **********************************************************************/
        wprintf(L" Data hashing");
        bResult = CryptHashData(hHash, // Дескриптор хэша
                                pbData, // Указатель на буфер с данными для хэширования
                                GetArraySize(pbData), // Длина буфера
                                0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 2.5 Получение значения хэша                                     *
         **********************************************************************/
        wprintf(L" Retrieving hash value");

        /**********************************************************************
         * Получение размера буфера для хэша                                   *
         **********************************************************************/
        bResult = CryptGetHashParam(hHash,       // Дескриптор хэша
                                    HP_HASHVAL,  // Признак значения хэша
                                    NULL,        // Указатель на буфер для значения хэша
                                    &dwHashSize, // Указатель на длину буфера
                                    0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }

        /**********************************************************************
         * Выделение памяти для буфера                                         *
         **********************************************************************/
        pbHash = (BYTE*)malloc(dwHashSize * sizeof(BYTE));
        bResult = (pbHash != NULL);

        if (!bResult) {
            wprintf(L" -> Failed, out of memory\n\n");
            break;
        }

        /**********************************************************************
         * Получение указателя на буфер с хэшем                                *
         **********************************************************************/
        bResult = CryptGetHashParam(hHash,       // Дескриптор хэша
                                    HP_HASHVAL,  // Признак значения хэша
                                    pbHash,      // Указатель на буфер для значения хэша
                                    &dwHashSize, // Указатель на длину буфера
                                    0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 3. Печать значения хэша                                         *
         **********************************************************************/
        wprintf(L"Hash value is: \n");
        for (i = 0; i < dwHashSize; ++i) {
            wprintf(L"%0.2X ", pbHash[i]);
            if ((i + 1) % 8 == 0) {
                wprintf(L"\n");
            }
        }
        wprintf(L"\nHash has been calculated successfully.\n\n");

        /**********************************************************************
         * Шаг 4. Подпись хэшированных данных                                  *
         **********************************************************************/
        wprintf(L"Signing data");

        /**********************************************************************
         * Получение размера буфера для подписи                                *
         **********************************************************************/
        bResult = CryptSignHashW(hHash,        // Дескриптор хэша
                                 AT_SIGNATURE, // Тип ключа -- для подписи
                                 NULL, 0,
                                 NULL,              // Указатель на буфер для подписи
                                 &dwSignatureSize); // Указатель на размер буфера

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }

        /**********************************************************************
         * Выделение памяти для буфера                                         *
         **********************************************************************/
        pbSignature = (BYTE*)malloc(dwSignatureSize * sizeof(BYTE));
        bResult = (pbSignature != NULL);

        if (!bResult) {
            wprintf(L" -> FAILED, out of memory\n\n");
            break;
        }

        /**********************************************************************
         * Получение указателя на буфер с подписью                             *
         **********************************************************************/
        bResult = CryptSignHashW(hHash,        // Дескриптор хэша
                                 AT_SIGNATURE, // Тип ключа -- для подписи
                                 NULL, 0,
                                 pbSignature,       // Указатель на буфер для подписи
                                 &dwSignatureSize); // Указатель на размер буфера

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 5. Печать значения подписи                                      *
         **********************************************************************/
        wprintf(L"\nSignature value is: \n");
        for (i = 0; i < dwSignatureSize; ++i) {
            wprintf(L"%0.2X ", pbSignature[i]);
            if ((i + 1) % 8 == 0) {
                wprintf(L"\n");
            }
        }
        wprintf(L"\nData has been signed successfully.\n\n");

        /**********************************************************************
         * Шаг 6. Освобождение дескриптора хэша и памяти                       *
         **********************************************************************/
        if (hHash) {
            CryptDestroyHash(hHash);
        }
        hHash = 0;

        free(pbHash);
        pbHash = NULL;

        /**********************************************************************
         * Шаг 7. Проверка подписи данных                                      *
         **********************************************************************/
        wprintf(L"Verifying signature...\n");

        /**********************************************************************
         * Шаг 7.1 Создание объекта хэша                                       *
         **********************************************************************/
        wprintf(L" Creating hash object");
        bResult = CryptCreateHash(hProv,                 // Дескриптор криптопровайдера
                                  ALG_GOST3411_2012_256, // Идентификатор алгоритма хэширования
                                  0, 0,
                                  &hHash); // Указатель на дескриптор хэша

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 7.2 Задание параметров функции хэширования                      *
         * Задание параметров функции хэширования не предусмотрено в           *
         * ГОСТ Р 34.11-2012.                                                  *
         **********************************************************************/

        /**********************************************************************
         * Шаг 7.3 Вычисление значения хэша                                    *
         **********************************************************************/
        wprintf(L" Data hashing");
        bResult = CryptHashData(hHash, // Дескриптор хэша
                                pbData, // Указатель на буфер с данными для хэширования
                                GetArraySize(pbData), // Длина буфера
                                0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 7.4 Получение значения хэша                                     *
         **********************************************************************/
        wprintf(L" Retrieving hash value");

        /**********************************************************************
         * Получение размера буфера для хэша                                   *
         **********************************************************************/
        bResult = CryptGetHashParam(hHash,       // Дескриптор хэша
                                    HP_HASHVAL,  // Признак значения хэша
                                    NULL,        // Указатель на буфер для значения хэша
                                    &dwHashSize, // Указатель на длину буфера
                                    0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }

        /**********************************************************************
         * Выделение памяти для буфера                                         *
         **********************************************************************/
        pbHash = (BYTE*)malloc(dwHashSize * sizeof(BYTE));
        bResult = (pbHash != NULL);

        if (!bResult) {
            wprintf(L" -> FAILED, out of memory\n\n");
            break;
        }

        /**********************************************************************
         * Получение указателя на буфер с хэшем                                *
         **********************************************************************/
        bResult = CryptGetHashParam(hHash,       // Дескриптор хэша
                                    HP_HASHVAL,  // Признак значения хэша
                                    pbHash,      // Указатель на буфер для значения хэша
                                    &dwHashSize, // Указатель на длину буфера
                                    0);

        if (!bResult) {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 7.5 Печать значения хэша                                        *
         **********************************************************************/
        wprintf(L"Hash value is: \n");
        for (i = 0; i < dwHashSize; ++i) {
            wprintf(L"%0.2X ", pbHash[i]);
            if ((i + 1) % 8 == 0) {
                wprintf(L"\n");
            }
        }
        wprintf(L"\nHash has been calculated successfully.\n\n");

        /**********************************************************************
         * Шаг 7.6 Получение дескриптора открытого ключа                       *
         **********************************************************************/
        wprintf(L"Getting public key");
        bResult = CryptGetUserKey(hProv,        // Дескриптор криптопровайдера
                                  AT_SIGNATURE, // Назначение ключа -- для подписи
                                  &hKey);       // Дескриптор ключа

        if (!bResult) {
            if (GetLastError() == NTE_NO_KEY) {
                wprintf(L" -> FAILED, key does not exist (0x%0.8x)\n\n", GetLastError());
            } else {
                wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
            }
            break;
        }
        wprintf(L" -> OK\n");

        /**********************************************************************
         * Шаг 7.7 Проверка подписи                                            *
         **********************************************************************/
        wprintf(L"Verifying signature");
        bResult = CryptVerifySignatureW(hHash,           // Дескриптор хэша
                                        pbSignature,     // Указатель на буфер с подписью
                                        dwSignatureSize, // Длина буфера
                                        hKey,            // Дескриптор ключа
                                        NULL, 0);

        if (bResult) {
            wprintf(L" -> OK\n");
            wprintf(L"\nSignature has been verified successfully.\n\n");
        } else if (GetLastError() == NTE_BAD_SIGNATURE) {
            wprintf(L" -> FAILED\n");
            wprintf(L"Signature verification has failed!\n\n");
        } else {
            wprintf(L" -> FAILED, error number: 0x%0.8x\n\n", GetLastError());
        }

        break;
    }

    /**********************************************************************
     * Шаг 8. Освобождение памяти                                          *
     **********************************************************************/
    /**********************************************************************
     * Шаг 8.1 Освобождение дескрипторов ключа и хэша                      *
     **********************************************************************/
    if (hKey) {
        CryptDestroyKey(hKey);
    }
    if (hHash) {
        CryptDestroyHash(hHash);
    }

    /**********************************************************************
     * Шаг 8.2 Освобождение контекста криптопровайдера                     *
     **********************************************************************/
    if (hProv) {
        CryptReleaseContext(hProv, 0);
    }

    /**********************************************************************
     * Шаг 8.3 Освобождение памяти                                         *
     **********************************************************************/
    free(pbHash);
    free(pbSignature);

    if (bResult) {
        wprintf(L"Test has been completed successfully. ");
    } else {
        wprintf(L"Test has failed. Error number: 0x%0.8x.", GetLastError());
    }
    return 0;
}