/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/************************************************************************
 * Для работы примера с ECDSA и RSA ключами нужно внести соответствующие *
 * изменения в Common.h и TokenPreparation.c                             *
 ************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>

int main(void) {
    EVP_PKEY* key;    // Описатель ключевой пары
    ENGINE* rtEngine; // rtengine
    BIO* inBio;       // Описатель потока ввода
    BIO* outBio;      // Описатель потока вывода

    CMS_ContentInfo* cms; // Описатель CMS структуры

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Получение ключевой пары                                                *
     *************************************************************************/
    printf("  get_key_pair...\n");
    key = get_key_pair();
    CHECK("  get_key_pair", key != NULL, unregister_engine);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("cms_encrypted.pem", "r");
    CHECK("  BIO_new_file", inBio != NULL, free_key);

    /*************************************************************************
     * Чтение CMS структуры из файла                                          *
     *************************************************************************/
    cms = PEM_read_bio_CMS(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_CMS", cms != NULL, free_in_bio);

    /*************************************************************************
     * Открытие потока вывода в файл                                          *
     *************************************************************************/
    outBio = BIO_new_file("decrypted_data.txt", "w");
    CHECK("  BIO_new_file", outBio != NULL, free_cms);

    /*************************************************************************
     * Расшифрование данных из CMS структуры                                  *
     *************************************************************************/
    r = CMS_decrypt(cms, key, NULL, NULL, outBio, 0);
    CHECK("  CMS_decrypt", r == 1, free_out_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_out_bio:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBio);
free_cms:

    /*************************************************************************
     * Освобождение CMS структуры                                             *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
free_key:

    /*************************************************************************
     * Освобождение описателя ключевой пары                                   *
     *************************************************************************/
    printf("  free_key_pair...\n");
    r = free_key_pair(key);
    CHECK_RELEASE("  free_key_pair", r == 0, errorCode);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
