/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

int main(void) {
    EVP_PKEY* key;     // Описатель ключевой пары
    ENGINE* rtEngine;  // rtengine
    EVP_PKEY_CTX* ctx; // Контекст генерации ключевой пары
    BIO* bio;          // Описатель потока ввода/вывода

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Создание контекста генерации ключевой пары ГОСТ Р 34.10-2012 256 бит   *
     *************************************************************************/

    /*************************************************************************
     * Для использования ГОСТ Р 34.10-2012 512 бит необходимо заменить        *
     * NID_id_GostR3410_2012_256 на NID_id_GostR3410_2012_512                 *
     *************************************************************************/
    ctx = EVP_PKEY_CTX_new_id(NID_id_GostR3410_2012_256, rtEngine);
    CHECK("  EVP_PKEY_CTX_new_id", ctx != NULL, unregister_engine);

    /*************************************************************************
     * Инициализация контекста                                                *
     *************************************************************************/
    r = EVP_PKEY_keygen_init(ctx);
    CHECK("  EVP_PKEY_keygen_init", r == 1, free_context);

    /*************************************************************************
     * Установка набора параметров ключевой пары                              *
     *************************************************************************/
    r = EVP_PKEY_CTX_ctrl_str(ctx, "paramset", "id-GostR3410-2001-CryptoPro-A-ParamSet");
    CHECK("  EVP_PKEY_CTX_ctrl_str", r > 0, free_context);

    /*************************************************************************
     * Генерация ключевой пары                                                *
     *************************************************************************/
    key = NULL;
    r = EVP_PKEY_keygen(ctx, &key);
    CHECK("  EVP_PKEY_keygen", r == 1, free_context);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    bio = BIO_new_file("key.pem", "w");
    CHECK("  BIO_new_file", bio != NULL, free_key);

    /*************************************************************************
     * Запись ключа в файл                                                    *
     *************************************************************************/
    r = PEM_write_bio_PrivateKey(bio, key, NULL, NULL, 0, NULL, NULL);
    CHECK("  PEM_write_bio_PrivateKey", r == 1, free_bio);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;

free_bio:
    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(bio);

free_key:
    /*************************************************************************
     * Освобождение описателя ключевой пары                                   *
     *************************************************************************/
    EVP_PKEY_free(key);
free_context:

    /*************************************************************************
     * Освобождение контекста                                                 *
     *************************************************************************/
    EVP_PKEY_CTX_free(ctx);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
