/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>

#include <Common.h>

size_t generateCek(unsigned char** cek) {
    EVP_CIPHER_CTX* ctx; // Контекст генерации ключа шифрования содержимого
    const EVP_CIPHER* cipher; // Описатель алгоритма шифрования
    size_t size;              // Длина ключа шифрования содержимого

    size_t result = 0; // Результирующее значение функции
    int r;             // Код возврата

    /*************************************************************************
     * Создание контекста генерации ключа шифрования содержимого              *
     *************************************************************************/
    ctx = EVP_CIPHER_CTX_new();
    CHECK("    EVP_CIPHER_CTX_new", ctx != NULL, exit);

    /*************************************************************************
     * Получение алгоритма шифрования                                         *
     *************************************************************************/
    cipher = EVP_get_cipherbynid(rt_eng_nid_gost28147_cfb);
    CHECK("    EVP_get_cipherbynid", cipher != NULL, free_context);

    /*************************************************************************
     * Инициализация контекста генерации ключа шифрования содержимого         *
     *************************************************************************/
    r = EVP_CipherInit_ex(ctx, cipher, NULL, NULL, NULL, -1);
    CHECK("    EVP_CipherInit_ex", r == 1, free_context);

    /*************************************************************************
     * Получение размера ключа шифрования содержимого                         *
     *************************************************************************/
    size = EVP_CIPHER_CTX_key_length(ctx);
    CHECK("    EVP_CIPHER_CTX_key_length", size > 0, free_context);

    /*************************************************************************
     * Выделение памяти для ключа шифрования содержимого                      *
     *************************************************************************/
    *cek = OPENSSL_secure_malloc(size);
    CHECK("    OPENSSL_secure_malloc", *cek != NULL, free_context);

    /*************************************************************************
     * Генерация ключа шифрования содержимого                                 *
     *************************************************************************/
    r = EVP_CIPHER_CTX_rand_key(ctx, *cek);
    CHECK("    EVP_CIPHER_CTX_rand_key", r == 1, free_cek);

    /*************************************************************************
     * Установка результирующего значения                                     *
     *************************************************************************/
    result = size;
    goto free_context;
free_cek:

    /*************************************************************************
     * Освобождение ключа шифрования содержимого                              *
     *************************************************************************/
    OPENSSL_secure_free(*cek);
free_context:

    /*************************************************************************
     * Освобождение контекста                                                 *
     *************************************************************************/
    EVP_CIPHER_CTX_free(ctx);
exit:
    return result;
}

int wrapKey(EVP_PKEY* key, unsigned char* cek, size_t cekSize) {
    EVP_PKEY_CTX* ctx; // Контекст шифрования

    unsigned char* wrappedKey; // Ключ шифрования содержимого
    size_t wrappedKeySize;     // Длина ключа шифрования содержимого
    FILE* out;                 // Описатель потока вывода

    size_t numberOfBytes; // Число записанных байт
    int r;                // Код возврата
    int errorCode = 1;    // Флаг ошибки

    /*************************************************************************
     * Создание контекста шифрования                                          *
     *************************************************************************/
    ctx = EVP_PKEY_CTX_new(key, NULL);
    CHECK("    EVP_PKEY_CTX_new", ctx != NULL, exit);

    /*************************************************************************
     * Инициализация контекста шифрования                                     *
     *************************************************************************/
    r = EVP_PKEY_encrypt_init(ctx);
    CHECK("    EVP_PKEY_encrypt_init", r == 1, free_context);

    /*************************************************************************
     * Определение размера зашифрованного ключа                               *
     *************************************************************************/
    r = EVP_PKEY_encrypt(ctx, NULL, &wrappedKeySize, cek, cekSize);
    CHECK("    EVP_PKEY_encrypt", r == 1, free_context);

    /*************************************************************************
     * Выделение памяти для зашифрованного ключа                              *
     *************************************************************************/
    wrappedKey = OPENSSL_malloc(wrappedKeySize);
    CHECK("    OPENSSL_malloc", wrappedKey != NULL, free_context);

    /*************************************************************************
     * Шифрование ключа                                                       *
     *************************************************************************/
    r = EVP_PKEY_encrypt(ctx, wrappedKey, &wrappedKeySize, cek, cekSize);
    CHECK("    EVP_PKEY_encrypt", r == 1, free_wrappedKey);

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    out = fopen("wrapped_key", "wb");
    CHECK("    fopen", out != NULL, free_wrappedKey);

    /*************************************************************************
     * Запись зашифрованного ключа в файл                                     *
     *************************************************************************/
    numberOfBytes = fwrite(wrappedKey, 1, wrappedKeySize, out);
    CHECK("    fwrite", numberOfBytes == wrappedKeySize, close_out_wrapped_key);

    /*************************************************************************
     * Установка признака успешного завершения подпрограммы                   *
     *************************************************************************/
    errorCode = 0;
close_out_wrapped_key:

    /*************************************************************************
     * Закрытие потока вывода зашифрованного ключа                            *
     *************************************************************************/
    r = fclose(out);
    CHECK_RELEASE("    fclose", r == 0, errorCode);
free_wrappedKey:

    /*************************************************************************
     * Освобождение зашифрованного ключа                                      *
     *************************************************************************/
    OPENSSL_free(wrappedKey);
free_context:

    /*************************************************************************
     * Освобождение контекста                                                 *
     *************************************************************************/
    EVP_PKEY_CTX_free(ctx);
exit:
    return errorCode;
}

int encryptFile(unsigned char* cek) {
    EVP_CIPHER_CTX* ctx;       // Контекст шифрования
    const EVP_CIPHER* cipher;  // Описатель алгоритма шифрования
    FILE* input;               // Описатель потока ввода
    FILE* output;              // Описатель потока вывода
    size_t numberOfBytes;      // Число прочитанных или записанных байт
    unsigned char* data;       // Данные для шифрования
    unsigned char* ciphertext; // Шифротекст
    unsigned char iv[8];       // Инициализирующий вектор
    int dataLen;               // Длина данных для шифрования
    int ciphertextLen;         // Длина шифротекста
    int sizeIv;                // Длина инициализируещего вектора
    int count;                 // Количество зашифрованных байт

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    input = fopen("test_data", "rb");
    CHECK("    fopen", input != NULL, exit);

    /*************************************************************************
     * Определение размера файла                                              *
     *************************************************************************/
    r = fseek(input, 0, SEEK_END);
    CHECK("    fseek", r == 0, close_input);
    dataLen = ftell(input);
    CHECK("    ftell", dataLen > 0, close_input);
    r = fseek(input, 0, SEEK_SET);
    CHECK("    fseek", r == 0, close_input);

    /*************************************************************************
     * Выделение памяти под данные для шифрования                             *
     *************************************************************************/
    data = OPENSSL_malloc(dataLen);
    CHECK("    OPENSSL_malloc", data != NULL, close_input);

    /*************************************************************************
     * Чтение данных для шифрования                                           *
     *************************************************************************/
    numberOfBytes = fread(data, 1, dataLen, input);
    CHECK("    fread", numberOfBytes == (size_t)dataLen, free_data);

    /*************************************************************************
     * Создание контекста шифрования                                          *
     *************************************************************************/
    ctx = EVP_CIPHER_CTX_new();
    CHECK("    EVP_CIPHER_CTX_new", ctx != NULL, free_data);

    /*************************************************************************
     * Получение алгоритма шифрования                                         *
     *************************************************************************/
    cipher = EVP_get_cipherbynid(rt_eng_nid_gost28147_cfb);
    CHECK("    EVP_get_cipherbynid", cipher != NULL, free_context);

    /*************************************************************************
     * Получение длины инициализируещего вектора                              *
     *************************************************************************/
    sizeIv = EVP_CIPHER_iv_length(cipher);
    CHECK("    EVP_CIPHER_iv_length", sizeIv == 8, free_context);

    /*************************************************************************
     * Генерация инициализируещего вектора                                    *
     *************************************************************************/
    r = RAND_bytes(iv, sizeIv);
    CHECK("    RAND_bytes", r == 1, free_context);

    /*************************************************************************
     * Инициализация шифрования                                               *
     *************************************************************************/
    r = EVP_EncryptInit_ex(ctx, cipher, NULL, cek, iv);
    CHECK("    EVP_EncryptInit_ex", r == 1, free_context);

    /*************************************************************************
     * Выделение памяти для шифротекста                                       *
     *************************************************************************/
    ciphertext = OPENSSL_malloc(dataLen + EVP_CIPHER_CTX_block_size(ctx) - 1);
    CHECK("    OPENSSL_malloc", ciphertext != NULL, free_context);

    /*************************************************************************
     * Шифрование данных                                                      *
     *************************************************************************/
    r = EVP_EncryptUpdate(ctx, ciphertext, &count, data, dataLen);
    CHECK("    EVP_EncryptUpdate", r == 1, free_ciphertext);
    ciphertextLen = count;

    /*************************************************************************
     * Завершение шифрования данных                                           *
     *************************************************************************/
    r = EVP_EncryptFinal_ex(ctx, ciphertext + count, &count);
    CHECK("    EVP_EncryptFinal_ex", r == 1, free_ciphertext);
    ciphertextLen += count;

    /*************************************************************************
     * Открытие поточного вывода в файл                                       *
     *************************************************************************/
    output = fopen("text_encrypted_with_cek", "wb");
    CHECK("    fopen", output != NULL, free_ciphertext);

    /*************************************************************************
     * Запись инициализирующего вектора в файл                                *
     *************************************************************************/
    numberOfBytes = fwrite(iv, 1, sizeIv, output);
    CHECK("    fwrite", numberOfBytes == (size_t)sizeIv, close_output);

    /*************************************************************************
     * Запись шифротекста в файл                                              *
     *************************************************************************/
    numberOfBytes = fwrite(ciphertext, 1, ciphertextLen, output);
    CHECK("    fwrite", numberOfBytes == (size_t)ciphertextLen, close_output);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
close_output:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    r = fclose(output);
    CHECK_RELEASE("    fclose", r == 0, errorCode);
free_ciphertext:

    /*************************************************************************
     * Освобождение шифротекста                                               *
     *************************************************************************/
    OPENSSL_free(ciphertext);
free_context:

    /*************************************************************************
     * Освобождение контекста шифрования                                      *
     *************************************************************************/
    EVP_CIPHER_CTX_free(ctx);
free_data:

    /*************************************************************************
     * Освобождение данных                                                    *
     *************************************************************************/
    OPENSSL_free(data);
close_input:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    r = fclose(input);
    CHECK_RELEASE("    fclose", r == 0, errorCode);
exit:
    return errorCode;
}

int main(void) {
    EVP_PKEY* key;    // Описатель открытого ключа
    ENGINE* rtEngine; // rtengine

    unsigned char* cek; // Ключ шифрования содержимого
    size_t cekSize;     // Длина ключа шифрования содержимого

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Выделение защищенной области памяти                                    *
     *************************************************************************/
    CRYPTO_secure_malloc_init(4096, 32);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Получение открытого ключа                                              *
     *************************************************************************/
    printf("  get_public_key...\n");
    key = get_public_key();
    CHECK("  get_public_key", key != NULL, unregister_engine);

    /*************************************************************************
     * Генерация ключа шифрования содержимого                                 *
     *************************************************************************/
    printf("  generateCek...\n");
    cekSize = generateCek(&cek);
    CHECK("  generateCek", cekSize > 0, free_key);

    /*************************************************************************
     * Шифрование данных и запись шифротекста в файл                          *
     *************************************************************************/
    printf("  encryptFile...\n");
    r = encryptFile(cek);
    CHECK("  encryptFile", r == 0, free_cek);

    /*************************************************************************
     * Шифрование ключа шифрования содержимого и его запись в файл            *
     *************************************************************************/
    printf("  wrapKey...\n");
    r = wrapKey(key, cek, cekSize);
    CHECK("  wrapKey", r == 0, free_cek);

    /*************************************************************************
     * Установка признака успешного завершения программы                      *
     *************************************************************************/
    errorCode = 0;
free_cek:

    /*************************************************************************
     * Освобождение ключа шифрования содержимого                              *
     *************************************************************************/
    OPENSSL_secure_free(cek);
free_key:

    /*************************************************************************
     * Освобождение описателя открытого ключа                                 *
     *************************************************************************/
    printf("  free_public_key...\n");
    r = free_public_key(key);
    CHECK_RELEASE("  free_public_key", r == 0, errorCode);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    CRYPTO_secure_malloc_done();
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
