/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/****************************************************************************
 * Для работы данного примера необходим сертификат, выписанный на ключевой   *
 * паре, которая будет использована в примере. О том, как выписать           *
 * сертификат на имеющуюся ключевую пару, можно узнать из описания к         *
 * использованию rtengine через инструменты командной строки OpenSSL         *
 * настоящего пакета SDK (openssl/samples/tool/README.md)                    *
 * --------------------------------------------------------------------------*
 * Для работы примера с ECDSA и RSA ключами нужно внести соответствующие     *
 * изменения в Common.h и TokenPreparation.c                                 *
 *****************************************************************************/

#include <assert.h>

#include <Common.h>

#include <openssl/cms.h>

int main(void) {
    ENGINE* rtEngine; // rtengine
    BIO* inBio;       // Описатель потока ввода
    BIO* outBio;      // Описатель потока вывода

    X509_STORE* certStore;  // Описатель хранилища сертификатов
    STACK_OF(X509) * certs; // Описатель контейнера сертификатов
    X509* cert;             // Описатель сертификата
    X509* caCert;           // Описатель сертификата удостоверяющего центра
    CMS_ContentInfo* cms;   // Описатель CMS структуры

    int r;             // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");
    /*************************************************************************
     * Инициализация OPENSSL_crypto                                           *
     *************************************************************************/
    r = OPENSSL_init_crypto(OPENSSL_INIT_NO_LOAD_CONFIG | OPENSSL_INIT_NO_ATEXIT, NULL);
    CHECK("  OPENSSL_init_crypto", r, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", r == 1, exit);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    r = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", r == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    r = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", r == 1, finalize_engine);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file(TEST_CERT_NAME, "r");
    CHECK("  BIO_new_file", inBio != NULL, unregister_engine);

    /*************************************************************************
     * Чтение сертификата                                                     *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_X509", cert != NULL, free_in_bio);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("test_trusted_ca.cer", "r");
    CHECK("  BIO_new_file", inBio != NULL, free_cert);

    /*************************************************************************
     * Чтение сертификата удостоверяющего центра                              *
     *************************************************************************/
    caCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_X509", caCert != NULL, free_cert);

    /*************************************************************************
     * Открытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("cms_signed.pem", "r");
    CHECK("  BIO_new_file", inBio != NULL, free_certCA);

    /*************************************************************************
     * Чтение CMS структуры                                                   *
     *************************************************************************/
    cms = PEM_read_bio_CMS(inBio, NULL, NULL, NULL);
    CHECK("  PEM_read_bio_CMS", cms != NULL, free_certCA);

    /*************************************************************************
     * Создание контейнера сертификатов                                       *
     *************************************************************************/
    certs = sk_X509_new_null();
    CHECK("  sk_X509_new_null", certs != NULL, free_cms);

    /*************************************************************************
     * Добавление сертификата в контейнер                                     *
     *************************************************************************/
    r = sk_X509_push(certs, cert);
    CHECK("  sk_X509_push", r == 1, free_sk_certs);

    /*************************************************************************
     * Создание хранилища сертификатов                                        *
     *************************************************************************/
    certStore = X509_STORE_new();
    CHECK("  X509_STORE_new", certStore != NULL, free_sk_certs);

    /*************************************************************************
     * Добавление сертификата удостоверяющего центра в хранилище              *
     *************************************************************************/
    r = X509_STORE_add_cert(certStore, caCert);
    CHECK("  X509_STORE_add_cert", r == 1, free_store);

    /*************************************************************************
     * Открытие потока вывода                                                 *
     *************************************************************************/
    outBio = BIO_new_file("content.txt", "w");
    CHECK("  BIO_new_file", outBio != NULL, free_store);

    /*************************************************************************
     * Проверка CMS                                                           *
     *************************************************************************/
    r = CMS_verify(cms, certs, certStore, NULL, outBio, CMS_BINARY);

    if (r == 1) {
        printf("  Signature is correct!\n");
        errorCode = 0;
    } else {
        printf("  Signature is wrong or some other error has occurred in CMS_verify()!\n");
        errorCode = 2;
    }

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBio);
free_store:

    /*************************************************************************
     * Освобождение хранилища сертификатов                                    *
     *************************************************************************/
    X509_STORE_free(certStore);
free_sk_certs:

    /*************************************************************************
     * Освобождение контейнера сертификатов                                   *
     *************************************************************************/
    sk_X509_free(certs);
free_cms:

    /*************************************************************************
     * Освобождение CMS                                                       *
     *************************************************************************/
    CMS_ContentInfo_free(cms);
free_certCA:

    /*************************************************************************
     * Освобождение сертификата удостоверяющего центра                        *
     *************************************************************************/
    X509_free(caCert);
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(cert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    r = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", r == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    r = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", r == 1, errorCode);
exit:
    OPENSSL_cleanup();
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
