/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Этот пример подписывает файл test_data.xml (любой xml файл с указанным *
 * названием) в режиме Enveloped и создает в out подписанный документ     *
 * signed_data.xml. Для хардварной версии примера используется ключевая   *
 * пара и файл hardware_test_cert.cer, созданные примером                 *
 * TokenPreparation                                                       *
 *************************************************************************/

#include <assert.h>

#include <commonXmldsig.h>

/*************************************************************************
 * Поддерживаемые виды каноникализации описаны в commonXmldsig.h          *
 * Для использования другого вида каноникализации следует заменить        *
 * XMLSEC_CANON_EXCL_C14N на желаемый                                     *
 *************************************************************************/
#define XMLSEC_CANON XMLSEC_CANON_EXCL_C14N // Определение каноникализации

xmlNodePtr xmlsec_create_enveloped_signature_node(xmlDocPtr doc, xmlSecTransformId c14nMethodId,
                                                  xmlSecTransformId signMethodId) {
    xmlNodePtr signatureNode; // Нодa для подписи
    xmlNodePtr refNode;       // Нодa для ссылки
    xmlNodePtr rvNode; // Указатель для проверки валидности операций с нодами

    /*************************************************************************
     * Создание пустой ноды для подписи                                       *
     *************************************************************************/
    signatureNode = xmlsec_create_signature_node(doc, c14nMethodId, signMethodId);
    CHECK("   xmlsec_create_signature_node", signatureNode != NULL, exit);

    /*************************************************************************
     * Создание <dsig:Reference/>                                             *
     * Для ключа длинной 512 бит следует использовать                         *
     * xmlSecOpenSSLTransformGostR3411_2012_512Id и самостоятельно            *
     * сгенерировать на токене ключ длинной 512 бит с ID, указанным в         *
     * g_keyPairIdGost2012RtEngine в Common.h                                 *
     *************************************************************************/
    refNode =
        xmlSecTmplSignatureAddReference(signatureNode, xmlSecOpenSSLTransformGostR3411_2012_256Id, NULL, NULL, NULL);
    CHECK("   xmlSecTmplSignatureAddReference", refNode != NULL, exit);

    /*************************************************************************
     * Создание <dsig:Transform/>                                             *
     *************************************************************************/
    rvNode = xmlSecTmplReferenceAddTransform(refNode, xmlSecTransformEnvelopedId);
    CHECK("   xmlSecTmplReferenceAddTransform", rvNode != NULL, exit);

    return signatureNode;

exit:
    return NULL;
}

int main() {
    ENGINE* rtEngine;       // rtengine
    FILE* outFile;          // Описатель потока вывода
    xmlSecKeyPtr keyPair;   // Описатель ключа
    xmlDocPtr doc;          // Описатель XML документа
    xmlNodePtr node;        // Описатель XML ноды для подписи
    unsigned char* outBuff; // Буфер для хранения подписанного файла
    int outSize;            // Размер подписанного файла
    int rv;                 // Код возврата
    int errorCode = 1;      // Флаг ошибки

    printf("Sample has started.\n\n");

    /*************************************************************************
     * Инициализация xmlsec                                                   *
     *************************************************************************/
    rv = xmlsec_init();
    CHECK("  xmlsec_init", rv == 0, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_load_engine();
    CHECK(" rt_eng_load_engine", rv == 1, finalize_xmlsec);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    rv = ENGINE_init(rtEngine);
    CHECK(" ENGINE_init", rv == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    rv = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK(" ENGINE_set_default", rv == 1, finalize_engine);

    /*************************************************************************
     * Считывание файла в структуру XML документа                             *
     * Если не используется пролог в XML документе, то можно явно указать     *
     * необходимую кодировку во втором параметре функции xmlReadFile,         *
     * например: doc = xmlReadFile("test_data.xml", "UTF-8", 0);              *
     *************************************************************************/
    doc = xmlReadFile("test_data.xml", NULL, 0);
    CHECK("  xmlReadFile", doc != NULL, unregister_engine);

    /*************************************************************************
     * Получение ключевой пары                                                *
     *************************************************************************/
    printf("  xmlsec_get_key_pair...\n");
    keyPair = xmlsec_get_key_pair();
    CHECK("  xmlsec_get_key_pair", keyPair != NULL, free_xmlDoc);

    /*************************************************************************
     * Добавление сертификата к ключу                                         *
     *************************************************************************/
    rv = xmlSecCryptoAppKeyCertLoad(keyPair, TEST_CERT_NAME, xmlSecKeyDataFormatCertPem);
    if (rv != 0) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("    xmlSecCryptoAppKeyCertLoad", rv == 0, free_xmlDoc);

    /*************************************************************************
     * Создание в XML документе ноды для подписи                              *
     * Для ключа длинной 512 бит следует использовать                         *
     * xmlSecOpenSSLTransformGostR3410_2012GostR3411_2012_512Id и             *
     * самостоятельно сгенерировать на токене ключ длинной 512 бит с ID,      *
     * указанным в g_keyPairIdGost2012RtEngine в Common.h                     *
     *************************************************************************/
    printf("  xmlsec_create_enveloped_signature_node...\n");
    node = xmlsec_create_enveloped_signature_node(doc, XMLSEC_CANON,
                                                  xmlSecOpenSSLTransformGostR3410_2012GostR3411_2012_256Id);
    if (!node) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("  xmlsec_create_enveloped_signature_node", node != NULL, free_xmlDoc);

    /*************************************************************************
     * Подпись                                                                *
     *************************************************************************/
    printf("  xmlsec_sign...\n");
    rv = xmlsec_sign(doc, keyPair);
    if (rv == 2) {
        xmlSecKeyDestroy(keyPair);
    }
    CHECK("  xmlsec_sign", rv == 0, free_xmlDoc);

    /*************************************************************************
     * Открытие файла для сохранения полученной подписи                       *
     *************************************************************************/
    outFile = fopen("signed_data.xml", "wb");
    CHECK("  fopen", outFile != NULL, free_xmlDoc);

    /*************************************************************************
     * Помещение подписанных данных в буфер для вывода                        *
     *************************************************************************/
    xmlDocDumpMemory(doc, &outBuff, &outSize);
    CHECK("  xmlDocDumpMemory", outBuff != NULL && outSize > 0, close_outFile);

    /*************************************************************************
     * Запись подписанных данных в поток вывода                               *
     *************************************************************************/
    rv = fprintf(outFile, "%s", outBuff);
    CHECK("  fprintf", rv == outSize, free_outBuff);

    errorCode = 0;
free_outBuff:

    /*************************************************************************
     * Освобождение памяти, выделенной под буфер для записи                   *
     *************************************************************************/
    xmlFree(outBuff);
close_outFile:

    /*************************************************************************
     * Закрытие файла                                                         *
     *************************************************************************/
    fclose(outFile);

/*************************************************************************
 * Деинициализация библиотеки pkcs11ecp                                   *
 *************************************************************************/
#ifdef HARDWARE_KEYS
    logout(NULL_PTR);
    rv = finalizeToken();
    CHECK_RELEASE("    finalizeToken", rv == 0, errorCode);
#endif
free_xmlDoc:

    /*************************************************************************
     * Удаление xmlDoc                                                        *
     *************************************************************************/
    xmlFreeDoc(doc);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    rv = ENGINE_finish(rtEngine);
    CHECK_RELEASE(" ENGINE_finish", rv == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_unload_engine();
    CHECK_RELEASE(" rt_eng_unload_engine", rv == 1, errorCode);
finalize_xmlsec:

    /*************************************************************************
     * Финализация xmlsec                                                     *
     *************************************************************************/
    xmlsec_final();
exit:

    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
