/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Этот пример верифицирует подписанный файл signed_data.xml, созданный в *
 * результате работы примера XMLDSigXadesBesSign.                         *
 * Версии Software и Hardware этого примера идентичны, разделение         *
 * сделано для удобства использования                                     *
 *************************************************************************/

#include <commonXmldsig.h>

int xmlsec_verify_xades_signing_certificate_v2(xmlNodePtr signatureNode) {
    xmlNodePtr objectNode;           // Нода для подписанного объекта
    xmlChar* x509CertificateData;    // Данные сертификата из ноды
    xmlChar* issuerSerialBase64Data; // Имя эмитента сертификата из ноды
    xmlChar* digestValueData;        // Значение хэш-суммы из ноды
    char* certSerial;                // Полученный серийный номер сертификата
    char* certIssuer;                // Полученное имя эмитента сертификата
    unsigned char* certDigest;       // Полученная хэш-сумма сертификата
    char* signedIssuer;              // Полученное имя эмитента из подписи
    char* signedSerial;              // Полученный серийный номер из подписи

    BIO* inBio;                           // Буфер для данных сертификата
    RT_ESS_ISSUER_SERIAL* issuerSerialV2; // Структура серийного номера эмитента
    GENERAL_NAME* issuerGeneralName; // Имя эмитента, полученное по его серийному номеру
    X509* cert;                      // Сертификат, полученный из буфера
    int errorCode = 1;               // Флаг ошибки

    /*************************************************************************
     * Получение значения сертификата из ноды подписи                         *
     *************************************************************************/
    x509CertificateData = xmlNodeGetContent(xmlSecFindNode(signatureNode, xmlSecNodeX509Certificate, xmlSecDSigNs));
    CHECK("   xmlNodeGetContent", x509CertificateData != NULL, exit);

    /*************************************************************************
     * Получение значения имени эмитента из ноды подписи, закодированного     *
     * в формате base64                                                       *
     *************************************************************************/
    issuerSerialBase64Data = xmlNodeGetContent(xmlSecFindNode(signatureNode, (const xmlChar*)"IssuerSerialV2",
                                                              (const xmlChar*)"http://uri.etsi.org/01903/v1.3.2#"));
    CHECK("   xmlNodeGetContent", issuerSerialBase64Data != NULL, free_x509_data);

    /*************************************************************************
     * Получение значения серийного номера из ноды подписи                    *
     *************************************************************************/
    issuerSerialV2 = xmlsec_get_decoded_issuer_serial_v2((const char*)issuerSerialBase64Data);
    CHECK("   xmlsec_get_decoded_issuer_serial_v2", issuerSerialV2 != NULL, free_signed_serial_data);

    /*************************************************************************
     * Получение имени эмитента из серийного номера                           *
     *************************************************************************/
    issuerGeneralName = sk_GENERAL_NAME_value(issuerSerialV2->issuer, 0);
    CHECK("   sk_GENERAL_NAME_value", issuerGeneralName != NULL, free_signed_serial_struct);

    /*************************************************************************
     * Перевод имени эмитента из X509_NAME в строку                           *
     *************************************************************************/
    signedIssuer = xmlsec_get_cert_issuer(issuerGeneralName->d.dirn);
    CHECK("   xmlsec_get_cert_issuer", signedIssuer != NULL, free_signed_serial_struct);

    /*************************************************************************
     * Перевод серийного номера из ASN.1 в строку                             *
     *************************************************************************/
    signedSerial = xmlsec_get_serial(issuerSerialV2->serial);
    CHECK("   xmlsec_get_serial", signedSerial != NULL, free_signed_issuer);

    /*************************************************************************
     * Поиск ноды подписанного объекта                                        *
     *************************************************************************/
    objectNode = xmlSecFindNode(signatureNode, (const xmlChar*)"Object", xmlSecDSigNs);
    CHECK("   xmlSecFindNode", objectNode != NULL, free_signed_serial);

    /*************************************************************************
     * Получение значения хэш-суммы из ноды объекта                           *
     *************************************************************************/
    digestValueData = xmlNodeGetContent(xmlSecFindNode(objectNode, (const xmlChar*)"DigestValue", xmlSecDSigNs));
    CHECK("   xmlNodeGetContent", digestValueData != NULL, free_signed_serial);

    /*************************************************************************
     * Создание буфера для данных сертификата и занесение в него данных       *
     *************************************************************************/
    inBio = BIO_new(BIO_s_mem());
    CHECK("   BIO_new", inBio != NULL, free_digest_data);

    BIO_puts(inBio, "-----BEGIN CERTIFICATE-----\n");
    BIO_puts(inBio, (char*)x509CertificateData);
    BIO_puts(inBio, "\n-----END CERTIFICATE-----\n");

    /*************************************************************************
     * Чтение буфера и получение сертификата из него                          *
     *************************************************************************/
    cert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("   PEM_read_bio_X509", cert != NULL, free_bio);

    /*************************************************************************
     * Получение серийного номера из сертификата и сравнение                  *
     *************************************************************************/
    certSerial = xmlsec_get_cert_serial(cert);
    CHECK("   xmlsec_get_cert_serial", certSerial != NULL, free_x509);

    CHECK("   xmlStrcmp", xmlStrcmp((const xmlChar*)certSerial, (const xmlChar*)signedSerial) == 0, free_serial);

    /*************************************************************************
     * Получение имени эмитента из сертификата и сравнение                    *
     *************************************************************************/
    certIssuer = xmlsec_get_cert_issuer(X509_get_issuer_name(cert));
    CHECK("   xmlsec_get_cert_issuer", certIssuer != NULL, free_serial);

    CHECK("   xmlStrcmp", xmlStrcmp((const xmlChar*)certIssuer, (const xmlChar*)signedIssuer) == 0, free_issuer);

    /*************************************************************************
     * Получение хэш-суммы сертификата и сравнение.                           *
     * Если хэширование проводилось функцией хэширования с длиной хэш-кода,   *
     * равной 512 бит, то замените md_gost12_256 на md_gost12_512             *
     *************************************************************************/
    certDigest = xmlsec_digest_cert(cert, "md_gost12_256");
    CHECK("   xmlsec_digest_cert", certDigest != NULL, free_issuer);

    CHECK("   xmlStrcmp", xmlStrcmp((const xmlChar*)certDigest, digestValueData) == 0, free_digest);

    errorCode = 0;

free_digest:
    OPENSSL_free(certDigest);
free_issuer:
    OPENSSL_free(certIssuer);
free_serial:
    OPENSSL_free(certSerial);
free_x509:
    X509_free(cert);
free_bio:
    BIO_free(inBio);
free_digest_data:
    xmlFree(digestValueData);
free_signed_serial:
    OPENSSL_free(signedSerial);
free_signed_issuer:
    OPENSSL_free(signedIssuer);
free_signed_serial_struct:
    RT_ESS_ISSUER_SERIAL_free(issuerSerialV2);
free_signed_serial_data:
    xmlFree(issuerSerialBase64Data);
free_x509_data:
    xmlFree(x509CertificateData);
exit:
    return errorCode;
}

int main() {
    ENGINE* rtEngine;             // rtengine
    xmlDocPtr doc;                // Описатель XML документа
    xmlNodePtr node;              // Описатель XML ноды для подписи
    xmlSecKeysMngrPtr keyManager; // Менеджер ключей
    xmlSecDSigCtxPtr dsigCtx;     // Описатель контекста для верификации
    int rv;                       // Код возврата
    int errorCode = 1;            // Флаг ошибки

    printf("Sample has started.\n\n");

    /*************************************************************************
     * Инициализация xmlsec                                                   *
     *************************************************************************/
    rv = xmlsec_init();
    CHECK("  xmlsec_init", rv == 0, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_load_engine();
    CHECK(" rt_eng_load_engine", rv == 1, finalize_xmlsec);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    rv = ENGINE_init(rtEngine);
    CHECK(" ENGINE_init", rv == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    rv = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK(" ENGINE_set_default", rv == 1, finalize_engine);

    /*************************************************************************
     * Считывание файла в структуру XML документа                             *
     * Если не используется пролог в XML документе, то можно явно указать     *
     * необходимую кодировку во втором параметре функции xmlReadFile,         *
     * например: doc = xmlReadFile("signed_data.xml", "UTF-8", 0);            *
     *************************************************************************/
    doc = xmlReadFile("signed_data.xml", NULL, 0);
    CHECK("  xmlReadFile", doc != NULL, unregister_engine);

    /*************************************************************************
     * Поиск XML ноды с подписью                                              *
     *************************************************************************/
    node = xmlSecFindNode(xmlDocGetRootElement(doc), xmlSecNodeSignature, xmlSecDSigNs);
    CHECK("  xmlSecFindNode", node != NULL, free_xmlDoc);

    /*************************************************************************
     * Создание менеджера ключей                                              *
     *************************************************************************/
    keyManager = xmlSecKeysMngrCreate();
    CHECK("  xmlSecKeysMngrCreate", keyManager != NULL, free_xmlDoc);

    /*************************************************************************
     * Инициализация менеджера ключей                                         *
     *************************************************************************/
    rv = xmlSecCryptoAppDefaultKeysMngrInit(keyManager);
    CHECK("  xmlSecCryptoAppDefaultKeysMngrInit", rv == 0, free_keyManager);

    /*************************************************************************
     * Добавление root сертификата в хранилище ключей                         *
     *************************************************************************/
    rv = xmlSecCryptoAppKeysMngrCertLoad(keyManager, "test_trusted_ca.cer", xmlSecKeyDataFormatPem,
                                         xmlSecKeyDataTypeTrusted);
    CHECK("  xmlSecCryptoAppKeysMngrCertLoadMemory", rv == 0, free_keyManager);

    /*************************************************************************
     * Создание контекста для верификации подписи                             *
     *************************************************************************/
    dsigCtx = xmlSecDSigCtxCreate(keyManager);
    CHECK("  xmlSecDSigCtxCreate", dsigCtx != NULL, free_keyManager);

    /*************************************************************************
     * Верификация                                                            *
     *************************************************************************/
    rv = xmlSecDSigCtxVerify(dsigCtx, node);
    CHECK("  xmlSecDSigCtxVerify", rv == 0, destroy_ctx);

    rv = xmlsec_verify_xades_signing_certificate_v2(node);
    CHECK("  xmlsec_verify_xades_signing_certificate_v2", rv == 0, destroy_ctx);

    errorCode = 0;

destroy_ctx:

    /*************************************************************************
     * Удаление контекста                                                     *
     *************************************************************************/
    xmlSecDSigCtxDestroy(dsigCtx);
free_keyManager:

    /*************************************************************************
     * Удаление менеджера ключей                                              *
     *************************************************************************/
    xmlSecKeysMngrDestroy(keyManager);
free_xmlDoc:

    /*************************************************************************
     * Удаление xmlDoc                                                        *
     *************************************************************************/
    xmlFreeDoc(doc);
unregister_engine:

    /*************************************************************************
     * Выгрузка rtengine из OpenSSL                                           *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    rv = ENGINE_finish(rtEngine);
    CHECK_RELEASE(" ENGINE_finish", rv == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_unload_engine();
    CHECK_RELEASE(" rt_eng_unload_engine", rv == 1, errorCode);
finalize_xmlsec:

    /*************************************************************************
     * Деинициализация xmlsec                                                 *
     *************************************************************************/
    xmlsec_final();
exit:

    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}
