/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/*************************************************************************
 * Добавление доверенной временной метки в Xades подпись в качестве       *
 * неподписанного атрибута первому найденному подписанту.                 *
 * Этот пример использует Xades подпись, созданную примером               *
 * XMLDSigXadesBesSign                                                    *
 *************************************************************************/

#include <assert.h>
#include <string.h>

#include <commonXmldsig.h>

#include <openssl/ts.h>
#include <openssl/x509.h>

/*************************************************************************
 * Поддерживаемые виды каноникализации описаны в commonXmldsig.h          *
 * Для использования другого вида каноникализации следует заменить        *
 * XMLSEC_CANON_C14N11 на желаемый                                        *
 *************************************************************************/
#define XMLSEC_CANON XMLSEC_CANON_EXCL_C14N // Определение каноникализации
#define XMLSEC_CANON_URI (const xmlChar*)"http://www.w3.org/2001/10/xml-exc-c14n#"
#define ts_req_version 1
#define ts_req_cert_req 1

TS_REQ* create_ts_request(xmlDocPtr doc, xmlNodePtr signature);
xmlNodePtr create_xades_t_node_tree(xmlNodePtr signature);

X509_STORE* create_cert_store();
STACK_OF(X509) * get_tsa_cert();
int add_tst_to_xades(xmlNodePtr signature, TS_RESP* response);

int main(void) {
    ENGINE* rtEngine;   // rtengine
    BIO* outBio = NULL; // Описатель потока вывода
    BIO* inBio = NULL;  // Описатель потока ввода

    xmlDocPtr doc;                     // Описатель XML документа
    xmlNodePtr signature;              // Описатель XML ноды подписи
    xmlNodePtr unsignedPropertiesNode; // Описатель XML ноды UnsignedProperties

    TS_REQ* request = NULL; // Описатель структуры запроса временной метки
    TS_RESP* response = NULL; // Описатель структуры ответа сервера с временной меткой
    TS_VERIFY_CTX* ctx = NULL; // Описатель контекста проверки ответа сервера с временной меткой

    FILE* outFile;          // Описатель потока вывода для Xades-T шаблона
    unsigned char* outBuff; // Буфер для хранения Xades-T шаблона
    int outSize;            // Размер Xades-T шаблона

    int rv;            // Код возврата
    int errorCode = 1; // Флаг ошибки

    printf("Sample has started.\n\n");

    /*************************************************************************
     * Инициализация xmlsec                                                   *
     *************************************************************************/
    rv = xmlsec_init();
    CHECK("  xmlsec_init", rv == 0, exit);

    /*************************************************************************
     * Загрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_load_engine();
    CHECK("  rt_eng_load_engine", rv == 1, finalize_xmlsec);

    /*************************************************************************
     * Получение rtengine                                                     *
     *************************************************************************/
    rtEngine = rt_eng_get0_engine();
    assert(rtEngine);

    /*************************************************************************
     * Инициализация rtengine                                                 *
     *************************************************************************/
    rv = ENGINE_init(rtEngine);
    CHECK("  ENGINE_init", rv == 1, unload_engine);

    /*************************************************************************
     * Установка rtengine реализацией по умолчанию                            *
     *************************************************************************/
    rv = ENGINE_set_default(rtEngine, ENGINE_METHOD_ALL - ENGINE_METHOD_RAND);
    CHECK("  ENGINE_set_default", rv == 1, finalize_engine);

    /*************************************************************************
     * Считывание файла в стукртуру XML документа                             *
     *************************************************************************/
    doc = xmlParseFile("signed_data.xml");
    CHECK("  xmlParseFile", doc != NULL, unregister_engine);

    /*************************************************************************
     * Поиск XML ноды с подписью                                              *
     *************************************************************************/
    signature = xmlSecFindNode(xmlDocGetRootElement(doc), xmlSecNodeSignature, xmlSecDSigNs);
    CHECK("  xmlSecFindNode", signature != NULL, free_xmlDoc);
    /*************************************************************************
     * Создание запроса временной метки                                       *
     *************************************************************************/
    request = create_ts_request(doc, signature);
    CHECK("  create_ts_request", request != NULL, free_xmlDoc);

    /*************************************************************************
     * Создание ноды для временной метки                                      *
     *************************************************************************/
    unsignedPropertiesNode = create_xades_t_node_tree(signature);
    CHECK("  create_xades_t_node_tree", unsignedPropertiesNode != NULL, free_ts_req);

    /*************************************************************************
     * Открытие поточного вывода из файла                                     *
     *************************************************************************/
    outBio = BIO_new_file("requestXadesT.tsq", "wb");
    CHECK("  BIO_new_file", outBio != NULL, free_ts_req);

    /*************************************************************************
     * Запись запроса временной метки                                         *
     *************************************************************************/
    rv = i2d_TS_REQ_bio(outBio, request);
    CHECK("  i2d_TS_REQ_bio", rv == 1, free_out_bio);

    BIO_flush(outBio);

    printf(
        "\n\nPlease create reply on the time stamp request (requestXadesT.tsq)\n"
        "(see item 13 in the openssl/samples/tool/README.md (responseXadesT.tsr is out parameter))\n"
        "and press Enter to continue.\n");
    fflush(stdout);
    getchar();

    /*************************************************************************
     * Создание контекста проверки временной метки из запроса                 *
     *************************************************************************/
    ctx = TS_REQ_to_TS_VERIFY_CTX(request, NULL);
    CHECK("  TS_REQ_to_TS_VERIFY_CTX", ctx != NULL, free_out_bio);

    /*************************************************************************
     * Выставление флага проверки подписи метки                               *
     *************************************************************************/
    TS_VERIFY_CTX_add_flags(ctx, TS_VFY_SIGNATURE);
    /*************************************************************************
     * Создание хранилища сертификатов и добавление его в контекст проверки   *
     *************************************************************************/
    rv = TS_VERIFY_CTX_set_store(ctx, create_cert_store()) != NULL;
    CHECK("  TS_VERIFY_CTX_set_store", rv == 1, free_ctx);

    /*************************************************************************
     * Добавление сертификата TSA в контекст проверки                         *
     *************************************************************************/
    rv = TS_VERIFY_CTS_set_certs(ctx, get_tsa_cert()) != NULL;
    CHECK("  TS_VERIFY_CTS_set_certs", rv == 1, free_ctx);

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    BIO_free_all(inBio);
    inBio = BIO_new_file("responseXadesT.tsr", "rb");
    CHECK("  BIO_new_file", inBio != NULL, free_ctx);

    /*************************************************************************
     * Чтение ответа сервера с временной меткой                               *
     *************************************************************************/
    response = d2i_TS_RESP_bio(inBio, NULL);
    CHECK("  d2i_TS_RESP_bio", response != NULL, free_in_bio);

    /*************************************************************************
     * Проверка ответа от сервера                                             *
     *************************************************************************/
    rv = TS_RESP_verify_response(ctx, response);
    CHECK("  TS_RESP_verify_response", rv == 1, free_ts_resp);

    /*************************************************************************
     * Добавление временной метки в качестве неподписанного атрибута Xades    *
     *************************************************************************/
    rv = add_tst_to_xades(signature, response);
    CHECK("  add_tst_to_xades", rv == 1, free_ts_resp);

    /*************************************************************************
     * Открытие поточного вывода                                              *
     *************************************************************************/
    outFile = fopen("signed_with_tst.xml", "wb");
    CHECK("  fopen", outFile != NULL, free_ts_resp);

    /*************************************************************************
     * Помещение подписанных данных в буфер для вывода                        *
     *************************************************************************/
    xmlDocDumpMemory(doc, &outBuff, &outSize);
    CHECK("  xmlDocDumpMemory", outBuff != NULL && outSize > 0, close_outFile);

    /*************************************************************************
     * Запись подписанных данных в поток вывода                               *
     *************************************************************************/
    rv = fprintf(outFile, "%s", outBuff);
    CHECK("  fprintf", rv == outSize, free_outBuff);

    errorCode = 0;

free_outBuff:

    /*************************************************************************
     * Освобождение памяти, выделенной под буфер для записи                   *
     *************************************************************************/
    xmlFree(outBuff);
close_outFile:

    /*************************************************************************
     * Закрытие поточного вывода                                              *
     *************************************************************************/
    fclose(outFile);
free_ts_resp:

    /*************************************************************************
     * Освобождение TS_RESP структуры                                         *
     *************************************************************************/
    TS_RESP_free(response);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
free_ctx:

    /*************************************************************************
     * Освобождение контекста проверки                                        *
     *************************************************************************/
    TS_VERIFY_CTX_free(ctx);
free_ts_req:
    /*************************************************************************
     * Освобождение TS_REQ структуры                                          *
     *************************************************************************/
    TS_REQ_free(request);
free_out_bio:

    /*************************************************************************
     * Закрытие потока вывода                                                 *
     *************************************************************************/
    BIO_free_all(outBio);
free_xmlDoc:

    /*************************************************************************
     * Удаление xmlDoc                                                        *
     *************************************************************************/
    xmlFreeDoc(doc);
unregister_engine:

    /*************************************************************************
     * Разрегистрация rtengine из OpenSSL                                     *
     *************************************************************************/
    ENGINE_unregister_pkey_asn1_meths(rtEngine);
    ENGINE_unregister_pkey_meths(rtEngine);
    ENGINE_unregister_digests(rtEngine);
    ENGINE_unregister_ciphers(rtEngine);
finalize_engine:

    /*************************************************************************
     * Деинициализация rtengine                                               *
     *************************************************************************/
    rv = ENGINE_finish(rtEngine);
    CHECK_RELEASE("  ENGINE_finish", rv == 1, errorCode);
unload_engine:

    /*************************************************************************
     * Выгрузка rtengine                                                      *
     *************************************************************************/
    rv = rt_eng_unload_engine();
    CHECK_RELEASE("  rt_eng_unload_engine", rv == 1, errorCode);
finalize_xmlsec:

    /*************************************************************************
     * Финализация xmlsec                                                     *
     *************************************************************************/
    xmlsec_final();
exit:
    if (errorCode) {
        printf("\n\nSample has failed. Some error has occurred.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }
    return errorCode;
}

unsigned char* canonicalize_node(xmlDocPtr doc, xmlNodePtr node, xmlSecTransformId canId, size_t* out_len) {
    xmlSecTransformCtxPtr ctx;          // Описатель xml контекста каноникализации
    xmlSecTransformPtr transform;       // Описатель структуры каноникализации
    xmlNodeSetPtr canonicalizedNode;    // Множество с нодами-комментариями
    xmlSecNodeSetPtr xmlsecWrappedNode; // Множество нод без нод-комментариев
    unsigned char* out = NULL;          // Буфер для каноникализированной ноды

    int rv; // Код возврата

    /*************************************************************************
     * Создание контекста каноникализации                                     *
     *************************************************************************/
    ctx = xmlSecTransformCtxCreate();
    CHECK("      xmlSecTransformCtxCreate", ctx != NULL, exit);

    /*************************************************************************
     * Создание трансформации и ее добавление к контексту каноникализации     *
     *************************************************************************/
    transform = xmlSecTransformCtxCreateAndAppend(ctx, canId);
    CHECK("      xmlSecTransformCtxCreateAndAppend", transform != NULL, destroy_ctx);

    /*************************************************************************
     * Создание ноды множества                                                *
     *************************************************************************/
    canonicalizedNode = xmlXPathNodeSetCreate(node);
    CHECK("      xmlXPathNodeSetCreate", canonicalizedNode != NULL, destroy_ctx);

    /*************************************************************************
     * Добавление множества нод                                               *
     *************************************************************************/
    xmlsecWrappedNode = xmlSecNodeSetCreate(doc, canonicalizedNode, xmlSecNodeSetTreeWithoutComments);
    CHECK("      xmlSecNodeSetCreate", xmlsecWrappedNode != NULL, destroy_ctx);

    /*************************************************************************
     * Каноникализация                                                        *
     *************************************************************************/
    rv = xmlSecTransformCtxXmlExecute(ctx, xmlsecWrappedNode);
    CHECK("      xmlSecTransformCtxXmlExecute", rv == 0, destroy_xmlsec_nodeset);

    *out_len = xmlSecBufferGetSize(ctx->result);
    CHECK("      xmlSecBufferGetSize", *out_len != 0, destroy_xmlsec_nodeset);

    out = OPENSSL_malloc(*out_len);
    CHECK("      OPENSSL_malloc", out != NULL, destroy_xmlsec_nodeset);

    memcpy(out, xmlSecBufferGetData(ctx->result), *out_len);

destroy_xmlsec_nodeset:
    xmlSecNodeSetDestroy(xmlsecWrappedNode);
destroy_ctx:
    xmlSecTransformCtxDestroy(ctx);
exit:
    return out;
}

xmlNodePtr create_xades_t_node_tree(xmlNodePtr signature) {
    xmlNodePtr qualifyingPropertiesNode = NULL;        // Описатель XML ноды QualifyingProperties
    xmlNodePtr unsignedPropertiesNode = NULL;          // Описатель XML ноды UnsignedProperties
    xmlNodePtr unsignedSignaturePropertiesNode = NULL; // Описатель XML ноды UnsignedSignatureProperties
    xmlNodePtr signatureTimeStampNode = NULL;          // Описатель XML ноды SignatureTimeStamp
    xmlNodePtr canonicalizationMethodNode = NULL;      // Описатель XML ноды CanonicalizationMethod
    xmlNsPtr dsNs;                                     // Пространство имен
    int errorCode = 1;                                 // Код ошибки

    /*************************************************************************
     * Поиск ноды QualifyingProperties                                        *
     *************************************************************************/
    qualifyingPropertiesNode = xmlSecFindNode(signature, (const xmlChar*)"QualifyingProperties",
                                              (const xmlChar*)"http://uri.etsi.org/01903/v1.3.2#");
    CHECK("    xmlSecFindNode", qualifyingPropertiesNode != NULL, exit);

    /*************************************************************************
     * Создание ноды UnsignedProperties                                       *
     *************************************************************************/
    unsignedPropertiesNode = xmlNewChild(qualifyingPropertiesNode, NULL, (const xmlChar*)"UnsignedProperties", NULL);
    CHECK("    xmlNewChild", unsignedPropertiesNode != NULL, exit);

    /*************************************************************************
     * Создание ноды UnsignedSignatureProperties                              *
     *************************************************************************/
    unsignedSignaturePropertiesNode =
        xmlNewChild(unsignedPropertiesNode, NULL, (const xmlChar*)"UnsignedSignatureProperties", NULL);
    CHECK("    xmlNewChild", unsignedSignaturePropertiesNode != NULL, exit);

    /*************************************************************************
     * Создание ноды SignatureTimeStamp                                       *
     *************************************************************************/
    signatureTimeStampNode =
        xmlNewChild(unsignedSignaturePropertiesNode, NULL, (const xmlChar*)"SignatureTimeStamp", NULL);
    CHECK("    xmlNewChild", signatureTimeStampNode != NULL, exit);

    /*************************************************************************
     * Создание ноды CanonicalizationMethod                                   *
     *************************************************************************/
    canonicalizationMethodNode =
        xmlNewChild(signatureTimeStampNode, NULL, (const xmlChar*)"CanonicalizationMethod", NULL);
    CHECK("    xmlNewChild", canonicalizationMethodNode != NULL, exit);

    /*************************************************************************
     * Создание простанства имен.                                             *
     *************************************************************************/
    dsNs = xmlNewNs(canonicalizationMethodNode, (const xmlChar*)"http://www.w3.org/2000/09/xmldsig#",
                    (const xmlChar*)"ds");
    CHECK("    xmlNewNs", dsNs != NULL, exit);

    /*************************************************************************
     * Повторное создание простанства имен, т.к. каждый новый вызов           *
     * xmlNewNs() затирает выставленные ранее простанства имен                *
     *************************************************************************/
    xmlSetNs(canonicalizationMethodNode, dsNs);
    /*************************************************************************
     * Выставление метода каноникализации                                     *
     *************************************************************************/
    xmlSetProp(canonicalizationMethodNode, (const xmlChar*)"Algorithm", XMLSEC_CANON_URI);

    errorCode = 0;

exit:
    return errorCode ? NULL : unsignedPropertiesNode;
}

TS_REQ* create_ts_request(xmlDocPtr doc, xmlNodePtr signature) {
    TS_MSG_IMPRINT* msgImprint; // Описатель структуры захэшированных канонизировванных данных подписи
    X509_ALGOR* algo;           // Описатель структуры алгоритма хэширования
    TS_REQ* req = NULL;         // Описатель структуры запроса
    const EVP_MD* md = NULL;    // Описатель структуры метода хэширования
    unsigned char* digest;      // Хэш от подписи
    unsigned char* signatureValueCanon; // Канонизированные данные ноды SignatureValue
    size_t signatureValueCanonLen; // Размер канонизированных данных ноды SignatureValue
    xmlNodePtr signatureValueNode; // Описатель XML ноды SignatureValue
    xmlSecTransformId canonicalizationId; // ID используемого алгоритма канонизации
    int digestLen;                        // Длина хэша
    uint64_t nonce = 0;                   // Случайное 64-битное число
    ASN1_INTEGER* asnNonce; // Описатель струткуры ASN1_INTEGER, представляющей случайное число

    int rv; // Код возврата

    /*************************************************************************
     * XMLDSig 2.0 по умолчанию рекомендует использовать алгоритм C14N20,     *
     * либо C14N11 в режиме совместимости с версией стандарта 1.1. Однако,    *
     * если обрабатывается уже готовый Xades шаблон, и в нем присутствует     *
     * SignatureTimeStamp, в CanonicalizationMethod которого указан другой    *
     * алгоритм, предпочтение нужно отдать ему.                               *
     *************************************************************************/
    canonicalizationId = XMLSEC_CANON;

    /*************************************************************************
     * Поиск XML ноды SignatureValue                                          *
     *************************************************************************/
    signatureValueNode = xmlSecFindNode(signature, xmlSecNodeSignatureValue, xmlSecDSigNs);
    CHECK("    xmlSecFindNode", signatureValueNode != NULL, exit);

    /*************************************************************************
     * Каноникализация ноды SignatureValue                                    *
     *************************************************************************/
    signatureValueCanon = canonicalize_node(doc, signatureValueNode, canonicalizationId, &signatureValueCanonLen);
    CHECK("    canonicalize_node", signatureValueCanon != NULL, exit);

    /*************************************************************************
     * Вычисление хэша от подписи                                             *
     *************************************************************************/
    md = EVP_get_digestbyname("md_gost12_256");
    CHECK("    EVP_get_digestbyname", md != NULL, free_canon_value);

    digestLen = EVP_MD_size(md);
    digest = OPENSSL_malloc(digestLen);
    CHECK("    OPENSSL_malloc", digest != NULL, free_canon_value);

    rv = EVP_Digest(signatureValueCanon, signatureValueCanonLen, digest, NULL, md, NULL);
    CHECK("    EVP_Digest", rv == 1, free_digest);

    /*************************************************************************
     * Создание структуры захэшированной канонизированой подписи              *
     *************************************************************************/
    msgImprint = TS_MSG_IMPRINT_new();
    CHECK("    TS_MSG_IMPRINT_new", msgImprint != NULL, free_digest);

    algo = X509_ALGOR_new();
    CHECK("    X509_ALGOR_new", algo != NULL, free_msg_imprint);

    X509_ALGOR_set_md(algo, md);

    rv = TS_MSG_IMPRINT_set_algo(msgImprint, algo);
    CHECK("    TS_MSG_IMPRINT_set_algo", rv == 1, free_algo);

    rv = TS_MSG_IMPRINT_set_msg(msgImprint, digest, digestLen);
    CHECK("    TS_MSG_IMPRINT_set_msg", rv == 1, free_algo);

    /*************************************************************************
     * Создание случайного числа                                              *
     *************************************************************************/
    rv = RAND_bytes((unsigned char*)&nonce, sizeof(nonce));
    CHECK("    RAND_bytes", rv == 1, free_algo);

    asnNonce = ASN1_INTEGER_new();
    CHECK("    ASN1_INTEGER_new", asnNonce != NULL, free_algo);

    rv = ASN1_INTEGER_set_uint64(asnNonce, nonce);
    CHECK("    ASN1_INTEGER_set_uint64", rv == 1, free_nonce);

    /*************************************************************************
     * Создание структуры TS_REQ                                              *
     *************************************************************************/
    req = TS_REQ_new();
    CHECK("    TS_REQ_new", req != NULL, free_nonce);

    rv = TS_REQ_set_version(req, ts_req_version);
    CHECK("    TS_REQ_set_version", rv == 1, free_req);

    rv = TS_REQ_set_msg_imprint(req, msgImprint);
    CHECK("    TS_REQ_set_msg_imprint", rv == 1, free_req);

    rv = TS_REQ_set_nonce(req, asnNonce);
    CHECK("    TS_REQ_set_nonce", rv == 1, free_req);

    rv = TS_REQ_set_cert_req(req, ts_req_cert_req);
    CHECK("    TS_REQ_set_cert_req", rv == 1, free_req);

    goto free_nonce;
free_req:

    /*************************************************************************
     * Освобождение структуры запроса                                         *
     *************************************************************************/
    TS_REQ_free(req);
    req = NULL;
free_nonce:

    /*************************************************************************
     * Освобождение nonce                                                     *
     *************************************************************************/
    ASN1_INTEGER_free(asnNonce);
free_algo:

    /*************************************************************************
     * Освобождение структуры алгоритма хэширования                           *
     *************************************************************************/
    X509_ALGOR_free(algo);
free_msg_imprint:

    /*************************************************************************
     * Освобождение структуры TS_MSG_IMPRINT                                  *
     *************************************************************************/
    TS_MSG_IMPRINT_free(msgImprint);
free_digest:

    /*************************************************************************
     * Освобождение хэша                                                      *
     *************************************************************************/
    OPENSSL_free(digest);
free_canon_value:

    /*************************************************************************
     * Освобождение канонизированных данных                                   *
     *************************************************************************/
    OPENSSL_free(signatureValueCanon);
exit:
    return req;
}

X509_STORE* create_cert_store() {
    BIO* inBio;               // Описатель потока ввода
    X509* caCert;             // Описатель сертификата удостоверяющего центра
    X509_STORE* store = NULL; // Описатель хранилища сертификатов

    int r; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_trusted_ca.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата удостоверяющего центра                              *
     *************************************************************************/
    caCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", caCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание хранилища сертификатов                                        *
     *************************************************************************/
    store = X509_STORE_new();
    CHECK("    X509_STORE_new", store != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата удостоверяющего центра в хранилище              *
     *************************************************************************/
    r = X509_STORE_add_cert(store, caCert);
    CHECK("    X509_STORE_add_cert", r == 1, free_store);

    goto free_cert;
free_store:

    /*************************************************************************
     * Освобождение хранилища сертификатов                                    *
     *************************************************************************/
    X509_STORE_free(store);
    store = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(caCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return store;
}

STACK_OF(X509) * get_tsa_cert() {
    BIO* inBio;                   // Описатель потока ввода
    STACK_OF(X509)* certs = NULL; // Описатель контейнера сертификатов
    X509* tsaCert;                // Описатель сертификата

    int rv; // Код возврата

    /*************************************************************************
     * Открытие поточного ввода из файла                                      *
     *************************************************************************/
    inBio = BIO_new_file("test_tsa_cert.cer", "rb");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    /*************************************************************************
     * Чтение сертификата                                                     *
     *************************************************************************/
    tsaCert = PEM_read_bio_X509(inBio, NULL, NULL, NULL);
    CHECK("    PEM_read_bio_X509", tsaCert != NULL, free_in_bio);

    /*************************************************************************
     * Создание контейнера сертификатов                                       *
     *************************************************************************/
    certs = sk_X509_new_null();
    CHECK("    sk_X509_new_null", certs != NULL, free_cert);

    /*************************************************************************
     * Добавление сертификата в контейнер                                     *
     *************************************************************************/
    rv = sk_X509_push(certs, tsaCert);
    CHECK("    sk_X509_push", rv == 1, free_sk_certs);

    goto free_in_bio;
free_sk_certs:

    /*************************************************************************
     * Освобождение контейнера сертификатов                                   *
     *************************************************************************/
    sk_X509_free(certs);
    certs = NULL;
free_cert:

    /*************************************************************************
     * Освобождение сертификата                                               *
     *************************************************************************/
    X509_free(tsaCert);
free_in_bio:

    /*************************************************************************
     * Закрытие потока ввода                                                  *
     *************************************************************************/
    BIO_free_all(inBio);
exit:
    return certs;
}

int add_tst_to_xades(xmlNodePtr signature, TS_RESP* response) {
    PKCS7* token;                   // Временная метка
    unsigned char* tokenBuf = NULL; // Закодированная временная метка
    int tokenBufLen;                // Длина закодированной временной метки
    unsigned char* encoded = NULL;  // Указатель на зашифрованную хеш-сумму
    unsigned int encodedLen;        // Длина зашифрованной хеш-суммы

    xmlNodePtr signatureTimeStampNode; // Описатель XML ноды SignatureTimeStamp
    xmlNodePtr encapsulatedTimeStamp;  // Описатель XML ноды EncapsulatedTimeStamp

    int rv = 0; // Код возврата

    /*************************************************************************
     * Получение временной метки из ответа сервера                            *
     *************************************************************************/
    token = TS_RESP_get_token(response);
    CHECK("    TS_RESP_get_token", token != NULL, exit);

    /*************************************************************************
     * DER кодирование временной метки                                        *
     *************************************************************************/
    tokenBufLen = i2d_PKCS7(token, &tokenBuf);
    CHECK("    i2d_PKCS7", tokenBufLen >= 0, exit);

    /*************************************************************************
     * Base64 кодирование временной метки                                     *
     *************************************************************************/
    encodedLen = base64_buffer_len(tokenBufLen) + 1;
    encoded = OPENSSL_malloc(encodedLen);
    CHECK("    OPENSSL_malloc", encoded != NULL, free_tokenbuf);

    rv = EVP_EncodeBlock(encoded, tokenBuf, tokenBufLen);
    CHECK("    EVP_EncodeBlock", (rv > 0) && ((unsigned int)rv < encodedLen), free_encoded);

    rv = 0;

    /*************************************************************************
     * Добавление временной метки в Xades в качестве неподписанного атрибута  *
     *************************************************************************/
    signatureTimeStampNode = xmlSecFindNode(signature, (const xmlChar*)"SignatureTimeStamp",
                                            (const xmlChar*)"http://uri.etsi.org/01903/v1.3.2#");
    CHECK("    xmlSecFindNode", signatureTimeStampNode != NULL, free_encoded);

    encapsulatedTimeStamp = xmlNewChild(signatureTimeStampNode, NULL, (const xmlChar*)"EncapsulatedTimeStamp", encoded);
    CHECK("    xmlNewChild", encapsulatedTimeStamp != NULL, free_encoded);

    rv = 1;
free_encoded:

    /*************************************************************************
     * Освобождение закодированной в Base64 временной метки                   *
     *************************************************************************/
    OPENSSL_free(encoded);
free_tokenbuf:

    /*************************************************************************
     * Освобождение закодированной временной метки                            *
     *************************************************************************/
    OPENSSL_free(tokenBuf);
exit:
    return rv;
}
