/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#pragma once

#ifdef _WIN32
    #include <Windows.h>
#endif
#include <stdio.h>

#include <openssl/engine.h>
#include <openssl/pem.h>

#ifdef HARDWARE_KEYS
    #include <rtpkcs11.h>
    #if defined(__unix__) || defined(__APPLE__)
        #include <win2nix.h>
    #endif
#endif

#ifdef __APPLE__
    #include <rtengine/engine.h>
#else
    #include <rutoken/engine/engine.h>
#endif

#define arrsize(arr) sizeof(arr) / sizeof(*arr)

#define CHECK(msg, expression, label) \
    do {                              \
        printf("%s", msg);            \
        if (!(expression)) {          \
            printf(" -> Failed\n");   \
            goto label;               \
        } else {                      \
            printf(" -> OK\n");       \
        }                             \
    } while (0)

#define CHECK_RELEASE(msg, expression, errorCode) \
    do {                                          \
        printf("%s", msg);                        \
        if (!(expression)) {                      \
            printf(" -> Failed\n");               \
            errorCode = 1;                        \
        } else {                                  \
            printf(" -> OK\n");                   \
        }                                         \
    } while (0)

#ifdef HARDWARE_KEYS
    /* Имя библиотеки PKCS#11 */
    #ifdef _WIN32
        /* Библиотека только для Рутокен ЭЦП, поддерживает алгоритмы ГОСТ и RSA */
        #define PKCS11ECP_LIBRARY_NAME "rtPKCS11ECP.dll"
    #endif
    #ifdef __unix__
        /* Библиотека только для Рутокен ЭЦП, поддерживает алгоритмы ГОСТ и RSA */
        #define PKCS11ECP_LIBRARY_NAME "librtpkcs11ecp.so"
    #endif
    #ifdef __APPLE__
        /* Библиотека только для Рутокен ЭЦП, поддерживает алгоритмы ГОСТ и RSA */
        #define PKCS11ECP_LIBRARY_NAME "rtpkcs11ecp.framework/rtpkcs11ecp"
    #endif

CK_FUNCTION_LIST_PTR g_functionList;
HMODULE g_pkcsModule;
CK_SESSION_HANDLE g_session;
CK_BYTE g_keyPairIdGost2012RtEngine[] = { "GOST R 34.10-2012 (256 bits) sample key pair ID for rtEngine samples" };
CK_BYTE g_keyPairIdEcdsaRtEngine[] = { "ECDSA sample key pair ID for rtEngine samples" };
CK_BYTE g_keyPairIdRsaRtEngine[] = { "RSA sample key pair ID for rtEngine samples" };

CK_SESSION_HANDLE initializeToken(CK_FUNCTION_LIST_PTR_PTR functionList, HMODULE* pcksModule) {
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList
    CK_C_INITIALIZE_ARGS initArgs; // Аргументы для инициализации библиотеки rtPKCS11ecp
    CK_ULONG slotsCount;           // Количество слотов
    CK_SESSION_HANDLE session;     // Описатель сессии
    CK_SLOT_ID_PTR slots = NULL; // Указатель на массив идентификаторов слотов

    CK_RV rv; // Код возврата

    initArgs.CreateMutex = NULL_PTR;
    initArgs.DestroyMutex = NULL_PTR;
    initArgs.LockMutex = NULL_PTR;
    initArgs.UnlockMutex = NULL_PTR;
    initArgs.pReserved = NULL_PTR;
    initArgs.flags = CKF_OS_LOCKING_OK;

    /*************************************************************************
     * Загрузка библиотеки rtPKCS11ecp                                        *
     *************************************************************************/
    *pcksModule = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK("      LoadLibrary", *pcksModule != NULL, exit);

    /*************************************************************************
     * Получение адреса функции C_GetFunctionList                             *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(*pcksModule, "C_GetFunctionList");
    CHECK("      GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получение списка функций                                               *
     *************************************************************************/
    rv = getFunctionList(functionList);
    CHECK("      getFunctionList", rv == CKR_OK, unload_pkcs11);

    /*************************************************************************
     * Инициализация библиотеки rtPkcs11ecp                                   *
     *************************************************************************/
    rv = (*functionList)->C_Initialize(&initArgs);
    CHECK("      C_Initialize", rv == CKR_OK, unload_pkcs11);

    /*************************************************************************
     * Получение количества активных слотов                                   *
     *************************************************************************/
    rv = (*functionList)->C_GetSlotList(CK_TRUE, NULL_PTR, &slotsCount);
    CHECK("      C_GetSlotList", rv == CKR_OK, finalize_pkcs11);

    CHECK("      Checking available tokens", slotsCount > 0, finalize_pkcs11);

    /*************************************************************************
     * Выделение памяти под список слотов                                     *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotsCount * sizeof(CK_SLOT_ID));
    CHECK("      malloc", slots != NULL, finalize_pkcs11);

    /*************************************************************************
     * Получение списка слотов                                                *
     *************************************************************************/
    rv = (*functionList)->C_GetSlotList(CK_TRUE, slots, &slotsCount);
    CHECK("      C_GetSlotList", rv == CKR_OK, free_slots);

    /*************************************************************************
     * Открытие сессии в первом слоте                                         *
     *************************************************************************/
    rv = (*functionList)->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL, NULL, &session);
    CHECK("      C_OpenSession", rv == CKR_OK, free_slots);

    /*************************************************************************
     * Освобождение слотов                                                    *
     *************************************************************************/
    free(slots);

    return session;
free_slots:

    /*************************************************************************
     * Освобождение слотов                                                    *
     *************************************************************************/
    free(slots);
finalize_pkcs11:

    /*************************************************************************
     * Деинициализация библиотеки                                             *
     *************************************************************************/
    rv = (*functionList)->C_Finalize(NULL_PTR);
    CHECK("      C_Finalize", rv == CKR_OK, unload_pkcs11);
unload_pkcs11:

    /*************************************************************************
     * Выгрузка библиотеки из памяти                                          *
     *************************************************************************/
    CHECK("      FreeLibrary", FreeLibrary(*pcksModule) != 0, exit);

exit:
    return CK_INVALID_HANDLE;
}

int finalizeToken() {
    CK_RV rv;          // Код возврата
    int errorCode = 0; // Флаг ошибки

    /*************************************************************************
     * Закрытие сессии в слоте                                                *
     *************************************************************************/
    rv = g_functionList->C_CloseSession(g_session);
    CHECK_RELEASE("    C_CloseSession", rv == CKR_OK, errorCode);

    /*************************************************************************
     * Деинициализация библиотеки                                             *
     *************************************************************************/
    rv = g_functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE("    C_Finalize", rv == CKR_OK, errorCode);

    /*************************************************************************
     * Выгрузка библиотеки из памяти                                          *
     *************************************************************************/
    CHECK_RELEASE("    FreeLibrary", FreeLibrary(g_pkcsModule) != 0, errorCode);

    return errorCode;
}

CK_OBJECT_HANDLE getObjectFromToken(CK_FUNCTION_LIST_PTR functionList, CK_SESSION_HANDLE session,
                                    CK_ATTRIBUTE* findObjectTemplate, CK_ULONG findObjectTemplateAttributeCount) {
    CK_OBJECT_HANDLE objectHandle; // Описатель объекта
    CK_ULONG objectsCount;         // Количество найденных объектов

    CK_OBJECT_HANDLE result = CK_INVALID_HANDLE; // Результирующее значение
    CK_RV rv;                                    // Код возврата
    int errorCode = 0;                           // Флаг ошибки

    /*************************************************************************
     * Инициализация функции поиска                                           *
     *************************************************************************/
    rv = functionList->C_FindObjectsInit(session, findObjectTemplate, findObjectTemplateAttributeCount);
    CHECK("      C_FindObjectsInit", rv == CKR_OK, exit);

    /*************************************************************************
     * Поиск объекта по шаблону                                               *
     *************************************************************************/
    rv = functionList->C_FindObjects(session, &objectHandle, 1, &objectsCount);
    CHECK_RELEASE("      C_FindObjects", (rv == CKR_OK) && (objectsCount > 0), errorCode);

    /*************************************************************************
     * Финализация функции поиска                                             *
     *************************************************************************/
    rv = functionList->C_FindObjectsFinal(session);
    CHECK("      C_FindObjectsFinal", rv == CKR_OK, exit);

    if (!errorCode) {
        result = objectHandle;
    }
exit:
    return result;
}

EVP_PKEY* getHardwareKeyPair() {
    CK_UTF8CHAR userPin[] = { "12345678" };    // PIN-код пользователя Рутокен
    CK_ULONG userPinLen = sizeof(userPin) - 1; // Длина PIN-кода пользователя Рутокен
    rt_eng_p11_session* wrappedSession;        // Описатель обёрнутой сессии
    CK_OBJECT_CLASS objectClass;               // Описатель типа объекта
    CK_OBJECT_HANDLE privateKeyHandle;         // Описатель закрытого ключа
    CK_OBJECT_HANDLE publicKeyHandle;          // Описатель открытого ключа
    EVP_PKEY* key = NULL;                      // Описатель ключевой пары
    CK_RV rv;                                  // Код возврата
    CK_ATTRIBUTE findObjectTemplate[2];        // Шаблон поиска ключей

    printf("    initializeToken...\n");
    g_session = initializeToken(&g_functionList, &g_pkcsModule);
    CHECK("    initializeToken", g_session != CK_INVALID_HANDLE, exit);

    wrappedSession = rt_eng_p11_session_wrap(g_functionList, g_session, 0, NULL);
    CHECK("    rt_eng_p11_session_wrap", wrappedSession != NULL, finalize_token);

    /*************************************************************************
     * Задание шаблона поиска открытого ключа                                 *
     *************************************************************************/
    /************************************************************************
     * Для использования ECDSA ключа необходимо заменить                    *
     * g_keyPairIdGost2012RtEngine на g_keyPairIdEcdsaRtEngine              *
     * -------------------------------------------------------------------- *
     * Для использования RSA ключа необходимо заменить                      *
     * g_keyPairIdGost2012RtEngine на g_keyPairIdRsaRtEngine                *
     ************************************************************************/
    objectClass = CKO_PUBLIC_KEY;
    findObjectTemplate[0].type = CKA_CLASS;
    findObjectTemplate[0].pValue = &objectClass;
    findObjectTemplate[0].ulValueLen = sizeof(objectClass);
    findObjectTemplate[1].type = CKA_ID;
    findObjectTemplate[1].pValue = &g_keyPairIdGost2012RtEngine;
    findObjectTemplate[1].ulValueLen = sizeof(g_keyPairIdGost2012RtEngine) - 1;

    printf("    getObjectFromToken...\n");
    publicKeyHandle = getObjectFromToken(g_functionList, g_session, findObjectTemplate,
                                         sizeof(findObjectTemplate) / sizeof(findObjectTemplate[0]));
    CHECK("    getObjectFromToken", publicKeyHandle != CK_INVALID_HANDLE, free_wrapped_session);

    /*************************************************************************
     * Аутентификация пользователя                                            *
     *************************************************************************/
    rv = g_functionList->C_Login(g_session, CKU_USER, userPin, userPinLen);
    CHECK("      C_Login", rv == CKR_OK, free_wrapped_session);

    /*************************************************************************
     * Задание шаблона поиска закрытого ключа                                 *
     *************************************************************************/
    objectClass = CKO_PRIVATE_KEY;

    printf("    getObjectFromToken...\n");
    privateKeyHandle = getObjectFromToken(g_functionList, g_session, findObjectTemplate,
                                          sizeof(findObjectTemplate) / sizeof(findObjectTemplate[0]));
    CHECK("    getObjectFromToken", privateKeyHandle != CK_INVALID_HANDLE, logout);

    key = rt_eng_p11_key_pair_wrap(wrappedSession, privateKeyHandle, publicKeyHandle);
    CHECK("    rt_eng_p11_key_pair_wrap", key != NULL, logout);

    /*************************************************************************
     * Освобождение обёрнутой сессии                                          *
     *************************************************************************/
    rt_eng_p11_session_free(wrappedSession);

    return key;
logout:
    /*************************************************************************
     * Сброс прав доступа                                                     *
     *************************************************************************/
    rv = g_functionList->C_Logout(g_session);
    CHECK("    C_Logout", rv == CKR_OK, finalize_token);
free_wrapped_session:

    /*************************************************************************
     * Освобождение обёрнутой сессии                                          *
     *************************************************************************/
    rt_eng_p11_session_free(wrappedSession);
finalize_token:

    finalizeToken();
exit:
    return NULL;
}

EVP_PKEY* getHardwarePublicKey() {
    rt_eng_p11_session* wrappedSession; // Описатель обёрнутой сессии
    CK_OBJECT_CLASS objectClass;        // Описатель типа объекта
    CK_OBJECT_HANDLE publicKeyHandle;   // Описатель открытого ключа
    EVP_PKEY* key;                      // Описатель ключевой пары
    CK_ATTRIBUTE findObjectTemplate[2]; // Шаблон поиска ключей

    printf("    initializeToken...\n");
    g_session = initializeToken(&g_functionList, &g_pkcsModule);
    CHECK("    initializeToken", g_session != CK_INVALID_HANDLE, exit);

    wrappedSession = rt_eng_p11_session_wrap(g_functionList, g_session, 0, NULL);
    CHECK("    rt_eng_p11_session_wrap", wrappedSession != NULL, finalize_token);

    /*************************************************************************
     * Задание шаблона поиска открытого ключа                                 *
     *************************************************************************/
    /************************************************************************
     * Для использования ECDSA ключа необходимо заменить                    *
     * g_keyPairIdGost2012RtEngine на g_keyPairIdEcdsaRtEngine              *
     * -------------------------------------------------------------------- *
     * Для использования RSA ключа необходимо заменить                      *
     * g_keyPairIdGost2012RtEngine на g_keyPairIdRsaRtEngine                *
     ************************************************************************/
    objectClass = CKO_PUBLIC_KEY;
    findObjectTemplate[0].type = CKA_CLASS;
    findObjectTemplate[0].pValue = &objectClass;
    findObjectTemplate[0].ulValueLen = sizeof(objectClass);
    findObjectTemplate[1].type = CKA_ID;
    findObjectTemplate[1].pValue = &g_keyPairIdGost2012RtEngine;
    findObjectTemplate[1].ulValueLen = sizeof(g_keyPairIdGost2012RtEngine) - 1;

    printf("    getObjectFromToken...\n");
    publicKeyHandle = getObjectFromToken(g_functionList, g_session, findObjectTemplate,
                                         sizeof(findObjectTemplate) / sizeof(findObjectTemplate[0]));
    CHECK("    getObjectFromToken", publicKeyHandle != CK_INVALID_HANDLE, free_wrapped_session);

    key = rt_eng_p11_key_pair_wrap(wrappedSession, NULL_PTR, publicKeyHandle);
    CHECK("    rt_eng_p11_key_pair_wrap", key != NULL, free_wrapped_session);

    /*************************************************************************
     * Освобождение обёрнутой сессии                                          *
     *************************************************************************/
    rt_eng_p11_session_free(wrappedSession);

    return key;
free_wrapped_session:

    /*************************************************************************
     * Освобождение обёрнутой сессии                                          *
     *************************************************************************/
    rt_eng_p11_session_free(wrappedSession);
finalize_token:

    finalizeToken();
exit:
    return NULL;
}

int logout(EVP_PKEY* key) {
    CK_RV rv;          // Код возврата
    int errorCode = 0; // Флаг ошибки

    /*************************************************************************
     * Сделать описатель ключевой пары недействительным                       *
     *************************************************************************/
    if (key) {
        rv = rt_eng_p11_key_pair_invalidate(key);
        CHECK_RELEASE("    rt_eng_p11_key_pair_invalidate", rv == 1, errorCode);
    }

    /*************************************************************************
     * Сброс прав доступа                                                     *
     *************************************************************************/
    rv = g_functionList->C_Logout(g_session);
    CHECK_RELEASE("    C_Logout", rv == CKR_OK, errorCode);

    return errorCode;
}

#else
EVP_PKEY* getSoftwareKeyPair() {
    BIO* inBio;           // Описатель потока ввода
    EVP_PKEY* key = NULL; // Описатель ключевой пары

    inBio = BIO_new_file("test_peer_private_key.pem", "r");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    key = PEM_read_bio_PrivateKey(inBio, NULL, NULL, NULL);

    BIO_free_all(inBio);
exit:
    return key;
}

EVP_PKEY* getSoftwarePublicKey() {
    BIO* inBio;           // Описатель потока ввода
    EVP_PKEY* key = NULL; // Описатель ключевой пары

    inBio = BIO_new_file("test_peer_public_key.pem", "r");
    CHECK("    BIO_new_file", inBio != NULL, exit);

    key = PEM_read_bio_PUBKEY(inBio, NULL, NULL, NULL);

    BIO_free_all(inBio);
exit:
    return key;
}
#endif

EVP_PKEY* get_key_pair() {
#ifdef HARDWARE_KEYS
    return getHardwareKeyPair();
#else
    return getSoftwareKeyPair();
#endif
}

EVP_PKEY* get_public_key() {
#ifdef HARDWARE_KEYS
    return getHardwarePublicKey();
#else
    return getSoftwarePublicKey();
#endif
}

int free_key_pair(EVP_PKEY* key) {
    int r = 0;
#ifdef HARDWARE_KEYS
    logout(key);
    r = finalizeToken();
#endif
    EVP_PKEY_free(key);
    return r;
}

int free_public_key(EVP_PKEY* key) {
    int r = 0;
#ifdef HARDWARE_KEYS
    r = finalizeToken();
#endif
    EVP_PKEY_free(key);
    return r;
}

#ifdef HARDWARE_KEYS
    #define TEST_CERT_NAME "hardware_test_cert.cer"
#else
    #define TEST_CERT_NAME "software_test_cert.cer"
#endif
