/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#include <assert.h>
#include <string.h>

#include <Common.h>
#include <openssl-internals.h>

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlmemory.h>

/*************************************************************************
 * Определение макросов, необходимых для правильной работы xmlsec         *
 *************************************************************************/
#define XMLSEC_NO_XSLT
#define XMLSEC_CRYPTO_OPENSSL
#define XMLSEC_NO_SIZE_T

#include <openssl/asn1t.h>
#include <openssl/x509v3.h>

#include <xmlsec/crypto.h>
#include <xmlsec/templates.h>
#include <xmlsec/xmldsig.h>
#include <xmlsec/xmlsec.h>
#include <xmlsec/xmltree.h>

#include <xmlsec/openssl/crypto.h>
#include <xmlsec/openssl/evp.h>

#include "openssl-internals.h"

// Copy of ESS_ISSUER_SERIAL from openssl sources
ASN1_SEQUENCE(RT_ESS_ISSUER_SERIAL) = { ASN1_SEQUENCE_OF(RT_ESS_ISSUER_SERIAL, issuer, GENERAL_NAME),
                                        ASN1_SIMPLE(RT_ESS_ISSUER_SERIAL, serial,
                                                    ASN1_INTEGER) } ASN1_SEQUENCE_END(RT_ESS_ISSUER_SERIAL)
    IMPLEMENT_ASN1_FUNCTIONS(RT_ESS_ISSUER_SERIAL)
#define XMLSEC_CANON_C14N xmlSecTransformInclC14NId
#define XMLSEC_CANON_C14N_WITH_COMMENTS xmlSecTransformInclC14NWithCommentsId
#define XMLSEC_CANON_C14N11 xmlSecTransformInclC14N11Id
#define XMLSEC_CANON_C14N11_WITH_COMMENTS xmlSecTransformInclC14N11WithCommentsId
#define XMLSEC_CANON_EXCL_C14N xmlSecTransformExclC14NId
#define XMLSEC_CANON_EXCL_C14N_WITH_COMMENTS xmlSecTransformExclC14NWithCommentsId

        int base64_buffer_len(int x) {
    return (x + 2) / 3 * 4;
}

unsigned char* xmlsec_base64_decode(const char* data, unsigned int* out_len) {
    BIO* base64Bio = NULL;         // Буфер формата base64
    BIO* sourceBio = NULL;         // Буфер ввода вывода
    int decodedLen;                // Длина декодированных данных
    unsigned char* decoded = NULL; // Декодированные данные

    /*************************************************************************
     * Создание буфера для хранения данных в формате base64                   *
     *************************************************************************/
    base64Bio = BIO_new(BIO_f_base64());
    CHECK("      BIO_new", base64Bio != NULL, exit);

    /*************************************************************************
     * Установка флага на запись в одну строку без перехода на новую          *
     *************************************************************************/
    BIO_set_flags(base64Bio, BIO_FLAGS_BASE64_NO_NL);

    /*************************************************************************
     * Cоздание буфера ввода длины                                            *
     *************************************************************************/
    sourceBio = BIO_new_mem_buf(data, -1);
    CHECK("      BIO_new_mem_buf", sourceBio != NULL, freeBase64Bio);

    /*************************************************************************
     * Создание цепочки обработки данных                                      *
     *************************************************************************/
    BIO_push(base64Bio, sourceBio);

    /*************************************************************************
     * Выделение памяти для декодированных данных                             *
     *************************************************************************/
    decoded = OPENSSL_malloc(strlen(data));
    CHECK("      OPENSSL_malloc", decoded != NULL, freeSourceBio);

    /*************************************************************************
     * Чтение данных из буфера                                                *
     *************************************************************************/
    decodedLen = BIO_read(base64Bio, decoded, (int)strlen(data));
    CHECK("      BIO_read", decodedLen > 0, freeDecoded);

    /*************************************************************************
     * Реаллокация декодированных данных                                      *
     *************************************************************************/
    decoded = OPENSSL_realloc(decoded, decodedLen);
    CHECK("      OPENSSL_realloc", decoded != NULL, freeDecoded);

    assert(out_len);
    *out_len = decodedLen;

    goto freeSourceBio;

freeDecoded:
    OPENSSL_free(decoded);
freeSourceBio:
    BIO_free(sourceBio);
freeBase64Bio:
    BIO_free(base64Bio);
exit:
    return decoded;
}

char* xmlsec_base64_encode(const unsigned char* data, unsigned int len) {
    BIO* base64Bio = NULL;    // Буфер формата base64
    BIO* encodedBio = NULL;   // Буфер ввода вывода
    char* encodedData = NULL; // Закодированные данные
    unsigned int encodedLen;  // Длина закодированных данных
    int rv;                   // Код ошибки

    /*************************************************************************
     * Создание буфера для хранения данных в формате base64                   *
     *************************************************************************/
    base64Bio = BIO_new(BIO_f_base64());
    CHECK("      BIO_new", base64Bio != NULL, exit);

    /*************************************************************************
     * Установка флага на запись в одну строку без перехода на новую          *
     *************************************************************************/
    BIO_set_flags(base64Bio, BIO_FLAGS_BASE64_NO_NL);

    /*************************************************************************
     * Создание буфера для хранения данных вывода                             *
     *************************************************************************/
    encodedBio = BIO_new(BIO_s_mem());
    CHECK("      BIO_new", encodedBio != NULL, freeBase64Bio);

    /*************************************************************************
     * Создание цепочки обработки данных                                      *
     *************************************************************************/
    BIO_push(base64Bio, encodedBio);

    /*************************************************************************
     * Запись данных в буфер ввода                                            *
     *************************************************************************/
    rv = BIO_write(base64Bio, data, len);
    CHECK("      BIO_write", rv >= 0, freeEncodedBio);

    /*************************************************************************
     * Окончание записи в буфер ввода                                         *
     *************************************************************************/
    rv = BIO_flush(base64Bio);
    CHECK("      BIO_flush", rv == 1, freeEncodedBio);

    /*************************************************************************
     * Получаем указатель на закодированные данные и их длину                 *
     *************************************************************************/
    encodedLen = BIO_get_mem_data(encodedBio, &encodedData);
    CHECK("      BIO_get_mem_data", encodedLen > 0, freeEncodedBio);

    /*************************************************************************
     * Создание дубликата закодированных данных                               *
     *************************************************************************/
    encodedData = OPENSSL_strndup(encodedData, encodedLen);
    CHECK("      OPENSSL_strndup", encodedData != NULL, freeEncodedBio);

freeEncodedBio:
    BIO_free(encodedBio);
freeBase64Bio:
    BIO_free(base64Bio);
exit:
    return encodedData;
}

RT_ESS_ISSUER_SERIAL* xmlsec_get_decoded_issuer_serial_v2(const char* base64EncodedDer) {
    RT_ESS_ISSUER_SERIAL* issuerSerial = NULL; // Структура идентификатора издателя
    unsigned char* decodedIssuerSerial = NULL; // Серийный номер в виде структуры ASN.1
    unsigned int decodedLen; // Длина декодированного серийного номера

    /*************************************************************************
     * Декодирование серийного номера                                         *
     *************************************************************************/
    decodedIssuerSerial = xmlsec_base64_decode(base64EncodedDer, &decodedLen);
    CHECK("      xmlsec_base64_decode", decodedIssuerSerial != NULL, exit);

    /*************************************************************************
     * Перевод серийного номера из DER в ASN.1 структуру                      *
     *************************************************************************/
    issuerSerial = d2i_RT_ESS_ISSUER_SERIAL(NULL, (const unsigned char**)&decodedIssuerSerial, decodedLen);
    CHECK("      d2i_RT_ESS_ISSUER_SERIAL", issuerSerial != NULL, free_decoded);

    /***************************************************************************
     * d2i_RT_ESS_ISSUER_SERIAL сдвигает buf, необходимо вернуться назад на len *
     ***************************************************************************/
    decodedIssuerSerial -= decodedLen;

free_decoded:
    OPENSSL_free(decodedIssuerSerial);
exit:
    return issuerSerial;
}

char* xmlsec_get_encoded_issuer_serial_v2(const X509* cert) {
    unsigned char* buf;   // Буфер для смены системы кодирования
    int len;              // Длина серийного номера
    char* encoded = NULL; // Серийный номер в формате base64
    int rv;               // Код возврата

    RT_ESS_ISSUER_SERIAL* issuerSerial = NULL; // Структура идентификатора издателя
    GENERAL_NAME* name = NULL;                 // Структура имени издателя

    /***************************************************************************
     * Создание структуры идентификатора издателя                               *
     ***************************************************************************/
    issuerSerial = RT_ESS_ISSUER_SERIAL_new();
    CHECK("      ESS_ISSUER_SERIAL_new", issuerSerial != NULL, exit);

    /***************************************************************************
     * Создание структуры для получения идентификатора издателя                 *
     ***************************************************************************/
    name = GENERAL_NAME_new();
    CHECK("      GENERAL_NAME_new", name != NULL, free_issuer_serial);

    /***************************************************************************
     * Получение идентификатора издателя                                        *
     ***************************************************************************/
    name->type = GEN_DIRNAME;
    name->d.dirn = X509_NAME_dup(X509_get_issuer_name(cert));
    CHECK("      X509_NAME_dup", name->d.dirn != NULL, free_name);

    /***************************************************************************
     * Запись идентификатора издателя в структуру                               *
     ***************************************************************************/
    rv = sk_GENERAL_NAME_push(issuerSerial->issuer, name);
    CHECK("      sk_GENERAL_NAME_push", rv != 0, free_name);
    name = NULL;

    /***************************************************************************
     * Запись серийного номера издателя в структуру                             *
     ***************************************************************************/
    ASN1_INTEGER_free(issuerSerial->serial);
    issuerSerial->serial = ASN1_INTEGER_dup(X509_get0_serialNumber(cert));
    CHECK("      ASN1_INTEGER_dup", issuerSerial->serial != NULL, free_name);

    /***************************************************************************
     * Получение длины серийного номера                                         *
     ***************************************************************************/
    len = i2d_RT_ESS_ISSUER_SERIAL(issuerSerial, NULL);
    CHECK("      i2d_ESS_ISSUER_SERIAL", len != 0, free_name);

    /***************************************************************************
     * Выделение памяти для буфера                                              *
     ***************************************************************************/
    buf = OPENSSL_malloc(len);
    CHECK("      OPENSSL_malloc", buf != NULL, free_name);

    /***************************************************************************
     * Перевод серийного номера в DER кодировку                                 *
     * i2d_RT_ESS_ISSUER_SERIAL сдвигает buf, необходимо вернуться назад на len *
     ***************************************************************************/
    len = i2d_RT_ESS_ISSUER_SERIAL(issuerSerial, &buf);
    CHECK("      i2d_ESS_ISSUER_SERIAL", len != 0, free_buf);
    buf -= len;

    /***************************************************************************
     * Перевод серийного номера в формат base64                                 *
     ***************************************************************************/
    encoded = xmlsec_base64_encode(buf, len);
    CHECK("      xmlsec_base64_encode", encoded != NULL, free_buf);

free_buf:
    OPENSSL_free(buf);
free_name:
    if (name) {
        GENERAL_NAME_free(name);
    }
free_issuer_serial:
    RT_ESS_ISSUER_SERIAL_free(issuerSerial);
exit:
    return encoded;
}

char* xmlsec_get_cert_issuer(X509_NAME* issuer) {
    BIO* cnBio;             // Буфер для ввода и вывода
    BUF_MEM* issuerNameBuf; // Имя эмитента
    char* issuerStr = NULL; // Строка с именем эмитента
    int rv;                 // Код возврата

    /*************************************************************************
     * Буфер для хранения данных с ввода и вывода                             *
     *************************************************************************/
    cnBio = BIO_new(BIO_s_mem());
    CHECK("      BIO_new_file", cnBio != NULL, exit);

    /*************************************************************************
     * Записать имя эмитента в буфер                                          *
     *************************************************************************/
    rv = X509_NAME_print_ex(cnBio, issuer, 0, XN_FLAG_RFC2253);
    CHECK("      X509_NAME_print_ex", rv != 0, exit);

    BIO_get_mem_ptr(cnBio, &issuerNameBuf);

    /*************************************************************************
     * Выделение памяти для имени эмитента и его копирование из буфера        *
     *************************************************************************/
    issuerStr = OPENSSL_zalloc(issuerNameBuf->length + 1);
    CHECK("      OPENSSL_zalloc", issuerStr != NULL, free_bio);

    memcpy(issuerStr, issuerNameBuf->data, issuerNameBuf->length);

free_bio:
    BIO_free_all(cnBio);
exit:
    return issuerStr;
}

char* xmlsec_get_serial(ASN1_INTEGER* serial) {
    BIGNUM* serialBN;       // Серийный номер в длинном числе
    char* serialStr = NULL; // Строка с серийным номером

    /*************************************************************************
     * Преобразовать серийный номер в большое число                           *
     *************************************************************************/
    serialBN = ASN1_INTEGER_to_BN(serial, NULL);
    CHECK("      ASN1_INTEGER_to_BN", serialBN != NULL, exit);

    /*************************************************************************
     * Конвертировать длинное число в строку                                  *
     *************************************************************************/
    serialStr = BN_bn2dec(serialBN);
    CHECK("      BN_bn2dec", serialStr != NULL, free_bn);

free_bn:
    BN_free(serialBN);
exit:
    return serialStr;
}

char* xmlsec_get_cert_serial(X509* cert) {
    ASN1_INTEGER* sn;       // Серийный номер
    char* serialStr = NULL; // Строка с серийным номером

    /*************************************************************************
     * Получить из сертификата серийный номер                                 *
     *************************************************************************/
    sn = X509_get_serialNumber(cert);
    CHECK("      X509_get_serialNumber", sn != NULL, exit);

    serialStr = xmlsec_get_serial(sn);
exit:
    return serialStr;
}

unsigned char* xmlsec_digest_cert(X509* cert, const char* mdName) {
    const EVP_MD* md;              // Алгоритм получения хеш-суммы
    unsigned char* digest = NULL;  // Указатель на хеш-сумму
    unsigned int digestLen;        // Длина хеш-суммы
    unsigned char* encoded = NULL; // Указатель на зашифрованную хеш-сумму
    unsigned int encodedLen;       // Длина зашифрованной хеш-суммы
    int rv;                        // Код возврата

    /*************************************************************************
     * Получение алгоритма для формирования хеш-суммы                         *
     *************************************************************************/
    md = EVP_get_digestbyname(mdName);
    CHECK("      EVP_get_digestbyname", md != NULL, exit);

    /*************************************************************************
     * Выделение памяти, необходимой под алгоритм формирования хеш-суммы      *
     *************************************************************************/
    digestLen = EVP_MD_size(md);
    digest = OPENSSL_malloc(digestLen);
    CHECK("      OPENSSL_malloc", digest != NULL, exit);

    /*************************************************************************
     * Хеширование сертифката                                                 *
     *************************************************************************/
    rv = X509_digest(cert, md, digest, &digestLen);
    CHECK("      X509_digest", rv != 0, free_digest);

    /*************************************************************************
     * Конвертирование к BASE64 длины полученной хеш-суммы и выделение        *
     * памяти под зашифрованные данные                                        *
     *************************************************************************/
    encodedLen = base64_buffer_len(digestLen) + 1;
    encoded = OPENSSL_malloc(encodedLen);
    CHECK("      OPENSSL_malloc", encoded != NULL, free_digest);

    /*************************************************************************
     * Шифрование хеш-суммы                                                   *
     *************************************************************************/
    rv = EVP_EncodeBlock(encoded, digest, digestLen);
    CHECK("      EVP_EncodeBlock", (rv > 0) && ((unsigned int)rv < encodedLen), free_encoded);

    goto free_digest;

free_encoded:
    OPENSSL_free(encoded);
    encoded = NULL;
free_digest:
    OPENSSL_free(digest);
exit:
    return encoded;
}

int xmlsec_final(void) {
    int rv;            // Код возврата
    int errorCode = 0; // Флаг ошибки

    /*************************************************************************
     * Деинициализация библиотеки xmlsec-crypto (в данных примерах в качестве *
     * xmlsec-crypto используется xmlsec-openssl)                             *
     *************************************************************************/
    rv = xmlSecCryptoShutdown();
    CHECK_RELEASE("    xmlSecCryptoShutdown", rv == 0, errorCode);

    /*************************************************************************
     * Деинициализация криптобиблиотеки (в данных примерах в качестве         *
     * криптобиблиотеки используется openssl)                                 *
     *************************************************************************/
    rv = xmlSecCryptoAppShutdown();
    CHECK_RELEASE("    xmlSecCryptoAppShutdown", rv == 0, errorCode);

    /*************************************************************************
     * Деинициализация библиотеки xmlsec                                      *
     *************************************************************************/
    rv = xmlSecShutdown();
    CHECK_RELEASE("    xmlSecShutdown", rv == 0, errorCode);

    /*************************************************************************
     * Деинициализация библиотеки libxml                                      *
     *************************************************************************/
    xmlCleanupParser();

    return errorCode;
}

int xmlsec_init(void) {
    int rv; // Код возврата

    /*************************************************************************
     * Инициализация библиотеки libxml                                        *
     *************************************************************************/
    xmlInitParser();

    LIBXML_TEST_VERSION
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);

    /*************************************************************************
     * Инициализация библиотеки xmlsec                                        *
     *************************************************************************/
    rv = xmlSecInit();
    CHECK("    xmlSecInit", rv == 0, exit);

    /*************************************************************************
     * Проверка версии xmlsec на совместимость                                *
     *************************************************************************/
    rv = xmlSecCheckVersion();
    CHECK("    xmlSecCheckVersion", rv == 1, exit);

    /*************************************************************************
     * Инициализация криптобиблиотеки (в данных примерах в качестве           *
     * криптобиблиотеки используется openssl)                                 *
     *************************************************************************/
    rv = xmlSecCryptoAppInit(NULL);
    CHECK("    xmlSecCryptoAppInit", rv == 0, exit);

    /*************************************************************************
     * Инизиализация xmlsec-crypto (в данных примерах в качестве              *
     * xmlsec-crypto используется xmlsec-openssl)                             *
     *************************************************************************/
    rv = xmlSecCryptoInit();
    CHECK("    xmlSecCryptoInit", rv == 0, exit);

    return 0;

exit:

    /*************************************************************************
     * Финализация xmlsec                                                     *
     *************************************************************************/
    xmlsec_final();

    return 1;
}

xmlSecKeyPtr xmlsec_get_key_pair(void) {
    EVP_PKEY* keyPair;     // Описатель ключа в формате EVP_PKEY
    xmlSecKeyPtr key;      // Описатель ключа в формате xmlSecKey
    xmlSecKeyDataPtr data; // Описатель данных ключа
    int rv;                // Код возврата

    /*************************************************************************
     * Получение ключа в формате EVP_PKEY                                     *
     *************************************************************************/
    keyPair = get_key_pair();
    CHECK("    get_key_pair", keyPair != NULL, exit);

    /*************************************************************************
     * Получение значения ключа в формате, используемом xmlsec                *
     *************************************************************************/
    data = xmlSecOpenSSLEvpKeyAdopt(keyPair);
    CHECK("    xmlSecOpenSSLEvpKeyAdopt", data != NULL, free_evpKey);

    /*************************************************************************
     * Создание нового ключа xmlsec                                           *
     *************************************************************************/
    key = xmlSecKeyCreate();
    CHECK("    xmlSecKeyCreate", key != NULL, free_xmlKeyData);

    /*************************************************************************
     * Задание созданному ключу нового значения, полученного ранее            *
     *************************************************************************/
    rv = xmlSecKeySetValue(key, data);
    CHECK("    xmlSecKeySetValue", rv == 0, free_xmlKey);

    return key;

free_xmlKey:

    /*************************************************************************
     * Удаление ключа в формате xmlSecKey                                     *
     *************************************************************************/
    xmlSecKeyDestroy(key);
free_xmlKeyData:

    /*************************************************************************
     * Удаление данных ключа                                                  *
     *************************************************************************/
    xmlSecKeyDataDestroy(data);
free_evpKey:

    /*************************************************************************
     * Удаление ключа в формате EVP_PKEY                                      *
     *************************************************************************/
    EVP_PKEY_free(keyPair);
exit:
    return NULL;
}

xmlNodePtr xmlsec_create_signature_node(xmlDocPtr doc, xmlSecTransformId c14nMethodId, xmlSecTransformId signMethodId) {
    xmlNodePtr signatureNode; // Описатель xml ноды для подписи
    xmlNodePtr keyInfoNode;   // Описатель xml ноды для информации о ключе
    xmlNodePtr keyDataNode;   // Описатель xml ноды для данных ключа
    xmlNodePtr rvNode; // Указатель для проверки валидности операций с нодами

    /*************************************************************************
     * Создание шаблона для подписи                                           *
     *************************************************************************/
    signatureNode = xmlSecTmplSignatureCreate(doc, c14nMethodId, signMethodId, NULL);
    CHECK("    xmlSecTmplSignatureCreate", signatureNode != NULL, exit);

    /*************************************************************************
     * Добавление в шаблон <dsig:KeyInfo/>                                    *
     *************************************************************************/
    keyInfoNode = xmlSecTmplSignatureEnsureKeyInfo(signatureNode, NULL);
    CHECK("    xmlSecTmplSignatureEnsureKeyInfo", keyInfoNode != NULL, free_sigNode);

    /*************************************************************************
     * Добавление в шаблон <dsig:X509Data/>                                   *
     *************************************************************************/
    keyDataNode = xmlSecTmplKeyInfoAddX509Data(keyInfoNode);
    CHECK("    xmlSecTmplKeyInfoAddX509Data", keyDataNode != NULL, free_sigNode);

    /*************************************************************************
     * Добавление в шаблон сформированной ноды для подписи                    *
     *************************************************************************/
    rvNode = xmlSecAddChildNode(xmlDocGetRootElement(doc), signatureNode);
    CHECK("   xmlSecAddChildNode", rvNode != NULL, free_sigNode);

    return signatureNode;

free_sigNode:

    /*************************************************************************
     * Удаление ноды для подписи                                              *
     *************************************************************************/
    xmlFreeNode(signatureNode);
exit:
    return NULL;
}

int xmlsec_sign(xmlDocPtr doc, xmlSecKeyPtr key) {
    xmlNodePtr node;          // Описатель xml ноды для подписи
    xmlSecDSigCtxPtr dsigCtx; // Описатель xml контекста для подписи
    int rv;                   // Код возврата
    int errorCode = 2;        // Флаг ошибки

    /*************************************************************************
     * Поиск xml ноды для подписи                                             *
     *************************************************************************/
    node = xmlSecFindNode(xmlDocGetRootElement(doc), xmlSecNodeSignature, xmlSecDSigNs);
    CHECK("    xmlSecFindNode", node != NULL, exit);

    /*************************************************************************
     * Создание контекста для обработки <dsig:Signature/>                     *
     *************************************************************************/
    dsigCtx = xmlSecDSigCtxCreate(NULL);
    CHECK("    xmlSecDSigCtxCreate", dsigCtx != NULL, exit);

    /*************************************************************************
     * Задание значения ключа                                                 *
     *************************************************************************/
    dsigCtx->signKey = key;
    errorCode = 0;

    /*************************************************************************
     * Подпись ноды                                                           *
     *************************************************************************/
    rv = xmlSecDSigCtxSign(dsigCtx, node);
    CHECK_RELEASE("    xmlSecDSigCtxSign", rv == 0, errorCode);

    /*************************************************************************
     * Удаление контекста для подписи                                         *
     *************************************************************************/
    xmlSecDSigCtxDestroy(dsigCtx);

exit:
    return errorCode;
}