/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Пример использования функций расширения компании "Актив"               *
 * стандарта PKCS#11:                                                     *
 *  - установление соединения с Rutoken в первом доступном слоте;         *
 *  - чтение поддерживаемых политик ПИН-кодов и проверка поддержки новой  *
 *    политики;                                                           *
 *  - запись новой политики ПИН-кодов;                                    *
 *  - демонстрация соблюдения политики ПИН-кодов (невозможности задания   *
 *    неподходящего ПИН-кода);                                            *
 *  - возврат к ПИН-коду по умолчанию.                                    *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/
#include <Common.h>

/*************************************************************************
 * Шаблон для поиска объекта с бинарными данными                          *
 *************************************************************************/
CK_ATTRIBUTE hardwareObjectTemplate[] = {
    { CKA_CLASS, &hardwareFeatureObject, sizeof(hardwareFeatureObject) }, // Класс - Hardware Feature Object
    { CKA_HW_FEATURE_TYPE, &hardwareFeaturePinPolicy,
      sizeof(hardwareFeaturePinPolicy) }, // Описание типа объекта, атрибуты которого будут выведены
    { CKA_VENDOR_USER_TYPE, &userTypeUser, sizeof(userTypeUser) } // Тип пользователя
};

/*************************************************************************
 * Шаблон для получения бинарных данных из объекта                        *
 *************************************************************************/
CK_ATTRIBUTE supportedPolicies = { CKA_VENDOR_SUPPORTED_PIN_POLICIES, NULL_PTR, 0 };

/*************************************************************************
 * Шаблон для передачи бинарных данных в объект                           *
 *************************************************************************/
CK_ATTRIBUTE pinPolicyValueTemplate[] = {
    /*{ CKA_MODIFIABLE, NULL_PTR, 0 },*/                    // Изменяемость политики
    /*{ CKA_VENDOR_PIN_POLICIES_DELETABLE, NULL_PTR, 0 },*/ // (!) Недоступный атрибут - всегда true
    /*{ CKA_VENDOR_PIN_POLICY_MIN_LENGTH, NULL_PTR, 0 },*/ // Минимальная длина ПИН-кода
    /*{ CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH, NULL_PTR, 0 },*/ // Глубина истории ПИН-кодов. При ненулевом значении этого
                                                              // атрибута C_SetPIN может возвращать CKR_PIN_IN_HISTORY
    /*{ CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE, &boolFalse, sizeof(boolFalse) },*/ // Разрешение использования
    { CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED, &attributeTrue, sizeof(attributeTrue) } //,         // Наличие цифр
    /*{ CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED, NULL_PTR, 0 },*/ // Наличие прописных букв
    /*{ CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED, &attributeTrue, sizeof(attributeTrue) },*/ // Наличие строчных букв
    /*{ CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED, NULL_PTR, 0 },*/ // Наличие специальных символов
    /*{ CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED, NULL_PTR, 0 }*/ // Наличие различных символов
};

/*************************************************************************
 * Преобразует значение атрибута политики к строке                        *
 *************************************************************************/
const char* pinPolicyToStr(CK_ULONG policy) {
    switch (policy) {
    case CKA_MODIFIABLE:
        return "CKA_MODIFIABLE";
    case CKA_VENDOR_PIN_POLICY_MIN_LENGTH:
        return "CKA_VENDOR_PIN_POLICY_MIN_LENGTH";
    case CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH:
        return "CKA_VENDOR_PIN_POLICY_HISTORY_DEPTH";
    case CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE:
        return "CKA_VENDOR_PIN_POLICY_ALLOW_DEFAULT_PIN_USAGE";
    case CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED:
        return "CKA_VENDOR_PIN_POLICY_DIGIT_REQUIRED";
    case CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED:
        return "CKA_VENDOR_PIN_POLICY_UPPERCASE_REQUIRED";
    case CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED:
        return "CKA_VENDOR_PIN_POLICY_LOWERCASE_REQUIRED";
    case CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED:
        return "CKA_VENDOR_PIN_POLICY_SPEC_CHAR_REQUIRED";
    case CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED:
        return "CKA_VENDOR_PIN_POLICY_DIFF_CHARS_REQUIRED";
    default:
        return "Unknown";
    }
}

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_FUNCTION_LIST_EXTENDED_PTR functionListEx; // Указатель на список функций расширения PKCS#11, хранящийся в
                                                  // структуре CK_FUNCTION_LIST_EXTENDED
    CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

    CK_SLOT_ID_PTR slots;         // Массив идентификаторов слотов
    CK_ULONG slotCount;           // Количество идентификаторов слотов в массиве
    CK_ULONG objectCount;         // Количество хэндлов объектов в массиве
    CK_OBJECT_HANDLE_PTR objects; // Массив найденных объектов

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int r; // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/

    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на              *
     * расширенные функции                                                    *
     *************************************************************************/
    getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
    CHECK(" GetProcAddress", getFunctionListEx != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на расширенные функции                *
     *************************************************************************/
    rv = getFunctionListEx(&functionListEx);
    CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    printf(" Slots available: %d\n", (int)slotCount);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);

    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    printf("\nAuthentication...\n");

    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Администратора                                *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_SO, SO_PIN, SO_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Authentication has been completed successfully.\n");

    /*************************************************************************
     * Найти объект на токене                                                 *
     *************************************************************************/
    printf("\nFinding hardware feature object...\n");

    r = findObjects(functionList, session, hardwareObjectTemplate, arraysize(hardwareObjectTemplate), &objects,
                    &objectCount);
    CHECK(" FindObjects", r == 0, logout);
    CHECK(" Check hardware feature object is found", objectCount != 0, logout);
    CHECK(" Check hardware feature object is single", objectCount == 1, logout);

    /*************************************************************************
     * Получить размер каждого атрибута найденного объекта                    *
     *************************************************************************/
    printf("\nGetting data from object...\n");

    printf(" Getting the size of each attribute from attributeValueTemplate...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], (CK_ATTRIBUTE_PTR)&supportedPolicies, 1);
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_objects);

    /*************************************************************************
     * Выделить память под каждый из атрибутов                                *
     *************************************************************************/
    printf(" Allocating memory for attributes...\n");
    supportedPolicies.pValue = (CK_VOID_PTR)malloc(supportedPolicies.ulValueLen);
    CHECK_AND_LOG_ONLY_ERROR(supportedPolicies.pValue != NULL, "Allocating memory for attributes failed",
                             free_attributeValues);

    /*************************************************************************
     * Получить значения атрибутов                                            *
     *************************************************************************/
    printf(" Getting the values of attributes...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], (CK_ATTRIBUTE_PTR)&supportedPolicies, 1);
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_attributeValues);

    printf("Data from object is gotten.\n");

    /*************************************************************************
     * Проверить поддержку задаваемой политики                                *
     *************************************************************************/
    CK_ULONG i;

    printf("\nChecking policy support...\n");
    for (i = 0; i < arraysize(pinPolicyValueTemplate); ++i) {
        int isSupported = 0;

        CK_ULONG j;

        for (j = 0; j < supportedPolicies.ulValueLen / sizeof(CK_ULONG); ++j) {
            if (pinPolicyValueTemplate[i].type == ((CK_ULONG_PTR)supportedPolicies.pValue)[j]) {
                isSupported = 1;
                break;
            }
        }

        printf(" ");
        CHECK_AND_LOG(pinPolicyToStr(pinPolicyValueTemplate[i].type), isSupported, "Policy is not supported.",
                      free_attributeValues);
    }
    printf("Policy support is checked.\n");

    /*************************************************************************
     * Очистить память, выделенную под атрибуты объекта                       *
     *************************************************************************/
    free(supportedPolicies.pValue);

    /*************************************************************************
     * Записать новую политику ПИН-кодов                                      *
     *************************************************************************/
    printf("\nSetting new pin policy...\n");

    rv = functionList->C_SetAttributeValue(session, objects[0], pinPolicyValueTemplate,
                                           arraysize(pinPolicyValueTemplate));
    CHECK_AND_LOG(" C_SetAttributeValue", rv == CKR_OK, rvToStr(rv), free_objects);

    printf("New pin policy is set.\n");

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
    free(objects);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
    printf("\nSwitching user...\n");

    rv = functionList->C_Logout(session);
    CHECK_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), close_session);

    /*************************************************************************
     * Закрыть открытую сессию пользователя в слоте                           *
     *************************************************************************/
    rv = functionList->C_CloseSession(session);
    CHECK_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Открыть RW сессию пользователя в первом доступном слоте                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Authentication has been completed successfully.\n");

    /*************************************************************************
     * Попытаться выставить слабый ПИН-код                                    *
     *************************************************************************/
    printf("\nChecking features...\n");

    rv = functionList->C_SetPIN(session, USER_PIN, USER_PIN_LEN, ALPHABETIC_USER_PIN, ALPHABETIC_USER_PIN_LEN);
    CHECK_AND_LOG(" C_SetPIN - alphabetic (CKR_INAPPROPRIATE_PIN)", rv == CKR_INAPPROPRIATE_PIN, rvToStr(rv), logout);

    rv = functionList->C_SetPIN(session, USER_PIN, USER_PIN_LEN, NEW_USER_PIN, NEW_USER_PIN_LEN);
    CHECK_AND_LOG(" C_SetPIN - numeric (CKR_OK)", rv == CKR_OK, rvToStr(rv), logout);

    rv = functionList->C_SetPIN(session, NEW_USER_PIN, NEW_USER_PIN_LEN, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_SetPIN - default", rv == CKR_OK, rvToStr(rv), logout);

    printf("Features is checked.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    printf("\nFinalizing... \n");
    goto logout;

    /*************************************************************************
     * Очистить память, выделенную под атрибуты объекта                       *
     *************************************************************************/
free_attributeValues:
    free(supportedPolicies.pValue);

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
free_objects:
    free(objects);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}