/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен ЭЦП при помощи библиотеки PKCS#11 на языке C   *
 *------------------------------------------------------------------------*
 * Удаление объектов, созданных предыдущими примерами:                    *
 *  - установление соединения с Рутокен ЭЦП в первом доступном слоте;     *
 *  - выполнение аутентификации Пользователя;                             *
 *  - поиск ключевой пары ГОСТ Р 34.10-2012 с длиной закрытого ключа      *
 *    256 бит и сертификата на Рутокен;                                   *
 *  - удаление найденных объектов;                                        *
 *  - сброс прав доступа Пользователя и закрытие соединения с Рутокен.    *
 *------------------------------------------------------------------------*
 * Пример удаляет объекты, созданные в памяти Рутокен примерами           *
 * CreateCSR-PKCS10-GOST34.10-2012-256 и                                  *
 * ImportCertificate-GOST34.10-2012-256                                   *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Шаблон для поиска открытого ключа ГОСТ Р 34.10-2012(256)               *
 *************************************************************************/
CK_ATTRIBUTE publicKeyTemplate[] = {
    { CKA_CLASS, &publicKeyObject, sizeof(publicKeyObject) }, // Класс - открытый ключ
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Открытый ключ является объектом токена
    { CKA_ID, &keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // Идентификатор ключевой пары
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_256,
      sizeof(keyTypeGostR3410_2012_256) }, // Тип ключа - ГОСТ Р 34.10-2012(256)
};

/*************************************************************************
 * Шаблон для поиска закрытого ключа ГОСТ Р 34.10-2012(256)               *
 *************************************************************************/
CK_ATTRIBUTE privateKeyTemplate[] = {
    { CKA_CLASS, &privateKeyObject, sizeof(privateKeyObject) }, // Класс - закрытый ключ
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Закрытый ключ является объектом токена
    { CKA_ID, &keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // Идентификатор ключевой пары
    { CKA_KEY_TYPE, &keyTypeGostR3410_2012_256,
      sizeof(keyTypeGostR3410_2012_256) }, // Тип ключа - ГОСТ Р 34.10-2012(256)
};

/*************************************************************************
 * Шаблон для поиска сертификата ключа подписи                            *
 *************************************************************************/
CK_ATTRIBUTE certificateTemplate[] = {
    { CKA_CLASS, &certificateObject, sizeof(certificateObject) }, // Класс - сертификат
    { CKA_TOKEN, &attributeTrue, sizeof(attributeTrue) }, // Сертификат является объектом токена
    { CKA_CERTIFICATE_TYPE, &certificateType, sizeof(certificateType) }, // Тип сертификата - X.509
    { CKA_ID, &keyPairIdGost2012_256_1, sizeof(keyPairIdGost2012_256_1) - 1 }, // Идентификатор искомого сертификата
    { CKA_CERTIFICATE_TYPE, &certificateType, sizeof(certificateType) }, // Тип сертификата - X.509
    { CKA_CERTIFICATE_CATEGORY, &tokenUserCertificate,
      sizeof(tokenUserCertificate) }, // Категория сертификата - пользовательский
};

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Указатель на массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_OBJECT_HANDLE_PTR publicKeys, privateKeys; // Массивы хэндлов объектов, соответствующих критериям поиска ключей
    CK_OBJECT_HANDLE_PTR certificates; // Массив хэндлов сертификатов, соответствующих критериям поиска сертификатов
    CK_ULONG publicKeysCount, privateKeysCount; // Количество хэндлов в массивах ключей
    CK_ULONG certificatesCount;                 // Количество сертификатов в массиве

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);

    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Удаление объектов                                                      *
     *************************************************************************/
    printf("\nDeleting objects...\n");

    /*************************************************************************
     * Поиск открытого ключа на токене                                        *
     *************************************************************************/
    printf(" Getting public key...\n");
    r = findObjects(functionList, session, publicKeyTemplate, arraysize(publicKeyTemplate), &publicKeys,
                    &publicKeysCount);
    CHECK(" findObjects", r == 0, logout);

    /*************************************************************************
     * Удаление открытого ключа                                               *
     *************************************************************************/
    if (publicKeysCount) {
        rv = functionList->C_DestroyObject(session, publicKeys[0]);
        CHECK_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), free_public_keys);
    } else {
        printf(" No public keys found\n");
    }

    /*************************************************************************
     * Поиск закрытого ключа на токене                                        *
     *************************************************************************/
    printf(" Getting private key...\n");
    r = findObjects(functionList, session, privateKeyTemplate, arraysize(privateKeyTemplate), &privateKeys,
                    &privateKeysCount);
    CHECK(" findObjects", r == 0, free_public_keys);

    /*************************************************************************
     * Удаление закрытого ключа                                               *
     *************************************************************************/
    if (privateKeysCount != 0) {
        rv = functionList->C_DestroyObject(session, privateKeys[0]);
        CHECK_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), free_private_keys);
    } else {
        printf(" No private keys found\n");
    }

    /*************************************************************************
     * Поиск сертификата на токене                                            *
     *************************************************************************/
    printf(" Getting certificate...\n");
    r = findObjects(functionList, session, certificateTemplate, arraysize(certificateTemplate), &certificates,
                    &certificatesCount);
    CHECK(" findObjects", r == 0, free_private_keys);

    /*************************************************************************
     * Удаление сертификата                                                   *
     *************************************************************************/
    if (certificatesCount) {
        rv = functionList->C_DestroyObject(session, certificates[0]);
        CHECK_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), free_certificates);
    } else {
        printf(" No certificates found\n");
    }

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память из-под объектов                                        *
     *************************************************************************/
free_certificates:
    free(certificates);

free_private_keys:
    free(privateKeys);

free_public_keys:
    free(publicKeys);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память из-под слотов                                          *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
