/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команды удаления объектов PKCS#11:                       *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - удаление ключей ГОСТ 34.12-2018 с длиной блока 128 бит (Кузнечик);  *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Пример удаляет все ключи, созданные в CreateForEncDec                  *
 * и CreateForSigVer                                                      *
 *************************************************************************/

#include <Common.h>

/************************************************************************
 * Шаблон для поиска симметричного ключа Кузнечик                        *
 ************************************************************************/
CK_ATTRIBUTE secKeyTmpl[] = {
    { CKA_ID, &secretKeyKuznechikId, sizeof(secretKeyKuznechikId) - 1 }, // Идентификатор ключа
    { CKA_KEY_TYPE, &keyTypeKuznechik, sizeof(keyTypeKuznechik) }        // Тип ключа - Кузнечик
};

/************************************************************************
 * Шаблон для поиска симметричного ключа Магма                           *
 ************************************************************************/
// CK_ATTRIBUTE secKeyTmpl[] =
//{
//	{ CKA_ID, &secretKeyMagmaId, sizeof(secretKeyMagmaId) - 1 }, // Идентификатор ключа
//	{ CKA_KEY_TYPE, &keyTypeMagma, sizeof(keyTypeMagma) }        // Тип ключа - Магма
// };

/************************************************************************
 * Шаблон для поиска симметричного ключа ГОСТ 28147-89                   *
 ************************************************************************/
// CK_ATTRIBUTE secKeyTmpl[] =
//{
//	{ CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },            // Идентификатор ключа
//	{ CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) } // Тип ключа - ГОСТ 28147-89
// };

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_OBJECT_HANDLE_PTR objects; // Массив найденных объектов
    CK_ULONG objectCount;         // Количество хэндлов объектов в массиве

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Получить массив хэндлов объектов, соответствующих критериям            *
     * поиска (производится в цикле)                                          *
     *************************************************************************/
    printf(" Getting secret keys...\n");
    r = findObjects(functionList, session, secKeyTmpl, arraysize(secKeyTmpl), &objects, &objectCount);
    CHECK(" findObjects", r == 0, logout);

    /*************************************************************************
     * Удалить все найденные объекты (производится в цикле)                   *
     *************************************************************************/
    if (objectCount > 0) {
        printf("\nDestroying objects...\n");

        for (i = 0; i < objectCount; ++i) {
            rv = functionList->C_DestroyObject(session, objects[i]);
            CHECK_AND_LOG(" C_DestroyObject", rv == CKR_OK, rvToStr(rv), free_objects);
        }

        printf("Objects have been destroyed successfully.\n");
    }

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
free_objects:
    free(objects);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
