/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд поиска бинарного объекта в памяти Рутокен:        *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - поиск бинарных данных в памяти Рутокен;                             *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Пример находит объект, созданный в CreateDataObject.                   *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Шаблон для поиска объекта с бинарными данными                          *
 *************************************************************************/
CK_ATTRIBUTE dataObjectTemplate[] = {
    { CKA_CLASS, &dataObject, sizeof(dataObject) }, // Класс - бинарные данные
    { CKA_APPLICATION, applicationLabel,
      sizeof(applicationLabel) - 1 }, // Описание приложения, которое записывает данные
    { CKA_LABEL, dataObjectLabel, sizeof(dataObjectLabel) - 1 } // Метка объекта с данными
};

/*************************************************************************
 * Шаблон для получения бинарных данных из объекта                        *
 *************************************************************************/
CK_ATTRIBUTE valueTemplate[] = { { CKA_VALUE, NULL_PTR, 0 } };

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_OBJECT_HANDLE_PTR objects; // Массив найденных объектов
    CK_ULONG objectCount;         // Количество хэндлов объектов в массиве

    CK_BYTE_PTR dataObjectValue; // Указатель на данные, хранящиеся внутри объекта

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Найти объект на токене                                                 *
     *************************************************************************/
    printf("\nFinding data object...\n");

    r = findObjects(functionList, session, dataObjectTemplate, arraysize(dataObjectTemplate), &objects, &objectCount);
    CHECK(" findObjects", r == 0, logout);
    CHECK(" Check object is found", objectCount != 0, logout);
    CHECK(" Check object is single", objectCount == 1, logout);

    /*************************************************************************
     * Получить и вывести данные                                              *
     *************************************************************************/
    printf("\nGetting data from object...\n");

    rv = functionList->C_GetAttributeValue(session, objects[0], valueTemplate, arraysize(valueTemplate));
    CHECK_AND_LOG(" C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_objects);

    dataObjectValue = (CK_BYTE_PTR)malloc(valueTemplate[0].ulValueLen);
    valueTemplate[0].pValue = dataObjectValue;

    rv = functionList->C_GetAttributeValue(session, objects[0], valueTemplate, arraysize(valueTemplate));
    CHECK_AND_LOG(" C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_dataObjectValue);

    printHex(dataObjectValue, valueTemplate[0].ulValueLen);

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под найденные бинарные данные              *
     *************************************************************************/
free_dataObjectValue:
    free(dataObjectValue);

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
free_objects:
    free(objects);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
