/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд инициализации Рутокен:                            *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - инициализация токена;                                               *
 *  - выполнение аутентификации Администратора;                           *
 *  - инициализация PIN-кода Пользователя;                                *
 *  - сброс прав доступа Администратора и закрытие соединения с Рутокен.  *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/

#include <Common.h>

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов всех доступных слотов
    CK_ULONG slotCount; // Количество идентификаторов всех доступных слотов в массиве

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME); //для Рутокен S на Windows надо использовать PKCS11_LIBRARY_NAME
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    printf("Initialization has been completed successfully.\n\n");

    /*************************************************************************
     * Инициализировать токен                                                 *
     *************************************************************************/
    rv = functionList->C_InitToken(slots[0], SO_PIN, SO_PIN_LEN, tokenLabel);
    CHECK_AND_LOG(" C_InitToken", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Выполнить аутентификацию Администратора                                *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_SO, SO_PIN, SO_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);

    /*************************************************************************
     * Инициализировать PIN-код Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_InitPIN(session, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_InitPIN", rv == CKR_OK, rvToStr(rv), logout);

    errorCode = 0;

    printf("\nFinalizing...\n");

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
