/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
﻿*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Пример использования функций расширения компании "Актив"               *
 * стандарта PKCS#11:                                                     *
 *  - установление соединения с Rutoken в первом доступном слоте;         *
 *  - вывод информации о токене через CKH_VENDOR_TOKEN_INFO.              *
 *------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                *
 *************************************************************************/
#include <Common.h>
#include <assert.h>

/*************************************************************************
 * Шаблон для поиска объекта с бинарными данными                          *
 *************************************************************************/

CK_ATTRIBUTE hardwareObjectTemplate[] = {
    { CKA_CLASS, &hardwareFeatureObject, sizeof(hardwareFeatureObject) }, // Класс - Hardware Feature Object
    { CKA_HW_FEATURE_TYPE, &hardwareFeatureTokenInfo,
      sizeof(hardwareFeatureTokenInfo) } // Описание типа объекта, атрибуты которого будут выведены
};

/*************************************************************************
 * Шаблон для получения бинарных данных из объекта                        *
 *************************************************************************/

CK_ATTRIBUTE attributeValueTemplate[] = {
    { CKA_VENDOR_EXTERNAL_AUTHENTICATION, NULL_PTR, 0 },
    { CKA_VENDOR_SECURE_MESSAGING_AVAILABLE, NULL_PTR, 0 },
    { CKA_VENDOR_CURRENT_SECURE_MESSAGING_MODE, NULL_PTR, 0 },
    { CKA_VENDOR_CURRENT_TOKEN_INTERFACE, NULL_PTR, 0 },
    { CKA_VENDOR_SUPPORTED_TOKEN_INTERFACE, NULL_PTR, 0 },
    { CKA_VENDOR_BIOMETRIC_AUTHENTICATION, NULL_PTR, 0 },
    { CKA_VENDOR_SUPPORTED_SECURE_MESSAGING_MODES, NULL_PTR, 0 },
    { CKA_VENDOR_SUPPORT_CUSTOM_PIN, NULL_PTR, 0 },
    { CKA_VENDOR_CUSTOM_ADMIN_PIN, NULL_PTR, 0 },
    { CKA_VENDOR_CUSTOM_USER_PIN, NULL_PTR, 0 },
    { CKA_VENDOR_SUPPORT_INTERNAL_TRUSTED_CERTS, NULL_PTR, 0 },
    { CKA_VENDOR_SUPPORT_FKC2, NULL_PTR, 0 },
};

/*************************************************************************
 * Выводит в консоль значение атрибута в человекочитаемой форме, если     *
 * невозможно распознать значение - выводит его в шестнадцатеричной форме *
 *************************************************************************/
void printAttributeVal(const CK_ATTRIBUTE attribute, // Атрибут, значение которого выводится
                       const size_t shiftSize // Кол-во знаков, на которое сдвигается новая строка
) {
#define PRINT_SHIFT                                    \
    for (size_t iter = shiftSize; iter != 0; --iter) { \
        printf(" ");                                   \
    }
#define SHIFT_IF_NEW_LINE(flag) \
    do {                        \
        if ((flag)) {           \
            PRINT_SHIFT         \
        } else {                \
            (flag) = 1;         \
        }                       \
    } while (0)

    if (attribute.ulValueLen == 0) {
        printf("VOID VALUE\n\n");
    }

    int isLineBreak = 0;

    switch (attribute.type) {
    /*************************************************************************
     * Семейство типов атрибута с бинарными значениями                        *
     *************************************************************************/
    case CKA_VENDOR_EXTERNAL_AUTHENTICATION:
    case CKA_VENDOR_SECURE_MESSAGING_AVAILABLE:
    case CKA_VENDOR_SUPPORT_CUSTOM_PIN:
    case CKA_VENDOR_CUSTOM_ADMIN_PIN:
    case CKA_VENDOR_CUSTOM_USER_PIN:
    case CKA_VENDOR_SUPPORT_INTERNAL_TRUSTED_CERTS:
    case CKA_VENDOR_SUPPORT_FKC2:
        if (*(CK_BBOOL*)attribute.pValue == CK_FALSE) {
            printf("FALSE\n\n");
            return;
        } else if (*(CK_BBOOL*)attribute.pValue == CK_TRUE) {
            printf("TRUE\n\n");
            return;
        }
        break;

    /*************************************************************************
     * Семейство типов атрибута со множественными значениями                  *
     *************************************************************************/
    case CKA_VENDOR_CURRENT_SECURE_MESSAGING_MODE:
        switch (*(CK_ULONG*)attribute.pValue) {
        case SECURE_MESSAGING_DEFAULT:
            printf("SECURE_MESSAGING_DEFAULT\n\n");
            return;
        case SECURE_MESSAGING_BUILT_IN:
            printf("SECURE_MESSAGING_BUILT_IN\n\n");
            return;
        case SECURE_MESSAGING_GOST:
            printf("SECURE_MESSAGING_GOST\n\n");
            return;
        case SECURE_MESSAGING_ENHANCED_GOST:
            printf("SECURE_MESSAGING_ENHANCED_GOST\n\n");
            return;
        case SECURE_MESSAGING_UNSUPPORTED:
            printf("SECURE_MESSAGING_UNSUPPORTED\n\n");
            return;
        }
        break;
    case CKA_VENDOR_CURRENT_TOKEN_INTERFACE:
        switch (*(CK_ULONG*)attribute.pValue) {
        case INTERFACE_TYPE_USB:
            printf("INTERFACE_TYPE_USB\n\n");
            return;
        case INTERFACE_TYPE_BT:
            printf("INTERFACE_TYPE_BT\n\n");
            return;
        case INTERFACE_TYPE_UART:
            printf("INTERFACE_TYPE_UART\n\n");
            return;
        case INTERFACE_TYPE_ISO:
            printf("INTERFACE_TYPE_ISO\n\n");
            return;
        case INTERFACE_TYPE_NFC_TYPE_B:
            printf("INTERFACE_TYPE_NFC_TYPE_B\n\n");
            return;
        case INTERFACE_TYPE_SPI:
            printf("INTERFACE_TYPE_SPI\n\n");
            return;
        case INTERFACE_TYPE_NFC_TYPE_A:
            printf("INTERFACE_TYPE_NFC_TYPE_A\n\n");
            return;
        case INTERFACE_TYPE_SD:
            printf("INTERFACE_TYPE_SD(DEPRECATED)\n\n");
            return;
        case INTERFACE_TYPE_UNKNOWN:
            printf("INTERFACE_TYPE_UNKNOWN\n\n");
            return;
        }
        break;
    case CKA_VENDOR_SUPPORTED_TOKEN_INTERFACE:
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_USB) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_USB\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_BT) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_BT\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_UART) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_UART\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_ISO) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_ISO\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_NFC_TYPE_B) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_NFC_TYPE_B\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_SPI) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_SPI\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_NFC_TYPE_A) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_NFC_TYPE_A\n");
        }
        if (*(CK_ULONG*)attribute.pValue & INTERFACE_TYPE_SD) {
            SHIFT_IF_NEW_LINE(isLineBreak);
            printf("INTERFACE_TYPE_SD\n");
        }
        if (isLineBreak) {
            printf("\n");
            return;
        }
        break;
    case CKA_VENDOR_BIOMETRIC_AUTHENTICATION:
        if (*(CK_ULONG*)attribute.pValue == BIOMETRIC_AUTHENTICATION_NOT_SUPPORTED) {
            printf("BIOMETRIC_AUTHENTICATION_NOT_SUPPORTED\n\n");
            return;
        }
        break;
    case CKA_VENDOR_SUPPORTED_SECURE_MESSAGING_MODES:
        for (size_t i = 0; i < attribute.ulValueLen / sizeof(CK_ULONG); ++i) {
            if (i != 0) {
                PRINT_SHIFT
            }
            switch (*((CK_ULONG*)attribute.pValue + i)) {
            case SECURE_MESSAGING_DEFAULT:
                printf("SECURE_MESSAGING_DEFAULT\n");
                break;
            case SECURE_MESSAGING_BUILT_IN:
                printf("SECURE_MESSAGING_BUILT_IN\n");
                break;
            case SECURE_MESSAGING_GOST:
                printf("SECURE_MESSAGING_GOST\n");
                break;
            case SECURE_MESSAGING_ENHANCED_GOST:
                printf("SECURE_MESSAGING_ENHANCED_GOST\n");
                break;
            case SECURE_MESSAGING_UNSUPPORTED:
                printf("SECURE_MESSAGING_UNSUPPORTED\n");
                break;
            default:
                printHexBuffer((CK_BYTE*)attribute.pValue + sizeof(CK_ULONG), sizeof(CK_ULONG), 0, 0, 0);
            }
            if (i == attribute.ulValueLen / sizeof(CK_ULONG) - 1) {
                printf("\n");
            }
        }
        return;
    }
    printHexBuffer(attribute.pValue, attribute.ulValueLen, 0, 0, 0);
    printf("\n");
    return;

#undef SHIFT_IF_NEW_LINE
#undef PRINT_SHIFT
}

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_FUNCTION_LIST_EXTENDED_PTR functionListEx; // Указатель на список функций расширения PKCS#11, хранящийся в
                                                  // структуре CK_FUNCTION_LIST_EXTENDED
    CK_C_EX_GetFunctionListExtended getFunctionListEx; // Указатель на функцию C_EX_GetFunctionListExtended

    CK_SLOT_ID_PTR slots;         // Массив идентификаторов слотов
    CK_ULONG slotCount;           // Количество идентификаторов слотов в массиве
    CK_ULONG objectCount;         // Количество хэндлов объектов в массиве
    CK_OBJECT_HANDLE_PTR objects; // Массив найденных объектов

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int r; // Код возврата для функций, возвращающих int

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/

    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на              *
     * расширенные функции                                                    *
     *************************************************************************/
    getFunctionListEx = (CK_C_EX_GetFunctionListExtended)GetProcAddress(module, "C_EX_GetFunctionListExtended");
    CHECK(" GetProcAddress", getFunctionListEx != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на расширенные функции                *
     *************************************************************************/
    rv = getFunctionListEx(&functionListEx);
    CHECK_AND_LOG(" Get function list extended", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    printf(" Slots available: %d\n", (int)slotCount);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL_PTR, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);

    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Открыть RW сессию в первом доступном слоте                             *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION | CKF_RW_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_slots);

    /*************************************************************************
     * Найти объект на токене                                                 *
     *************************************************************************/
    printf("\nFinding hardware feature object...\n");

    r = findObjects(functionList, session, hardwareObjectTemplate, arraysize(hardwareObjectTemplate), &objects,
                    &objectCount);
    CHECK(" FindObjects", r == 0, close_session);
    CHECK(" Check hardware feature object is found", objectCount != 0, close_session);

    /*************************************************************************
     * Получить размер каждого атрибута найденного объекта                    *
     *************************************************************************/
    printf("\nGetting data from object...\n");

    const char* Attributes[] = {
        " CKA_VENDOR_EXTERNAL_AUTHENTICATION:            ", " CKA_VENDOR_SECURE_MESSAGING_AVAILABLE:         ",
        " CKA_VENDOR_CURRENT_SECURE_MESSAGING_MODE:      ", " CKA_VENDOR_CURRENT_TOKEN_INTERFACE:            ",
        " CKA_VENDOR_SUPPORTED_TOKEN_INTERFACE:          ", " CKA_VENDOR_BIOMETRIC_AUTHENTICATION:           ",
        " CKA_VENDOR_SUPPORTED_SECURE_MESSAGING_MODES:   ", " CKA_VENDOR_SUPPORT_CUSTOM_PIN:                 ",
        " CKA_VENDOR_CUSTOM_ADMIN_PIN:                   ", " CKA_VENDOR_CUSTOM_USER_PIN:                    ",
        " CKA_VENDOR_SUPPORT_INTERNAL_TRUSTED_CERTS:     ", " CKA_VENDOR_SUPPORT_FKC2:                       "
    };

    assert(arraysize(Attributes) == arraysize(attributeValueTemplate));

    printf(" Getting the size of each attribute from attributeValueTemplate...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], attributeValueTemplate,
                                           arraysize(attributeValueTemplate));
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_objects);

    /*************************************************************************
     * Выделить память под каждый из атрибутов                                *
     *************************************************************************/
    CK_ULONG i;

    printf(" Allocating memory for attributes...\n");
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        attributeValueTemplate[i].pValue = (CK_VOID_PTR)malloc(attributeValueTemplate[i].ulValueLen);
        CHECK_AND_LOG_ONLY_ERROR(attributeValueTemplate[i].pValue != NULL, "Allocating memory for attributes failed",
                                 free_attributeValues);
    }

    /*************************************************************************
     * Получить значения атрибутов                                            *
     *************************************************************************/
    printf(" Getting the values of attributes...\n");
    rv = functionList->C_GetAttributeValue(session, objects[0], attributeValueTemplate,
                                           arraysize(attributeValueTemplate));
    CHECK_AND_LOG("  C_GetAttributeValue", rv == CKR_OK, rvToStr(rv), free_attributeValues);
    printf("\n");

    /*************************************************************************
     * Вывести значения атрибутов                                             *
     *************************************************************************/
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        printf("%s", Attributes[i]);
        printAttributeVal(attributeValueTemplate[i], strlen(Attributes[i]));
    }

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под атрибуты объекта
     *************************************************************************/
free_attributeValues:
    for (i = 0; i < arraysize(attributeValueTemplate); ++i) {
        free(attributeValueTemplate[i].pValue);
    }

    /*************************************************************************
     * Очистить память, выделенную под объекты                                *
     *************************************************************************/
free_objects:
    free(objects);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}