/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C       *
 *------------------------------------------------------------------------*
 * Использование команд выработки имитовставки на ключе ГОСТ 34.12-2018   *
 * с длиной блока 128 бит (Кузнечик):                                     *
 *  - установление соединения с Рутокен в первом доступном слоте;         *
 *  - выполнение аутентификации Пользователя;                             *
 *  - выработка имитовставки для сообщения на демонстрационном ключе;     *
 *  - проверка имитовставки на демонстрационном ключе;                    *
 *  - сброс прав доступа Пользователя на Рутокен и закрытие соединения    *
 *    с Рутокен.                                                          *
 *------------------------------------------------------------------------*
 * Пример использует объекты, созданные в памяти Рутокен примером         *
 * CreateForSigVer.                                                       *
 *************************************************************************/

#include <Common.h>

/*************************************************************************
 * Данные для выработки имитовставки                                      *
 *************************************************************************/
CK_BYTE data[] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };

/*************************************************************************
 * Шаблон для поиска симметричного ключа Кузнечик                         *
 *************************************************************************/
CK_ATTRIBUTE secKeyTemplate[] = {
    { CKA_ID, &secretKeyKuznechikId, sizeof(secretKeyKuznechikId) - 1 }, // Идентификатор ключа
    { CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },            // Класс - секретный ключ
    { CKA_KEY_TYPE, &keyTypeKuznechik, sizeof(keyTypeKuznechik) },       // Тип ключа - Кузнечик
    { CKA_SIGN, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для подписи
    { CKA_VERIFY, &attributeTrue, sizeof(attributeTrue) }, // Ключ предназначен для проверки подписи
};

/*************************************************************************
 * Шаблон для поиска симметричного ключа Магма                            *
 *************************************************************************/
// CK_ATTRIBUTE secKeyTemplate[] =
//{
//	{ CKA_ID, &secretKeyMagmaId, sizeof(secretKeyMagmaId) - 1 }, // Идентификатор ключа
//	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },    // Класс - секретный ключ
//	{ CKA_KEY_TYPE, &keyTypeMagma, sizeof(keyTypeMagma) },       // Тип ключа - Магма
//	{ CKA_SIGN, &attributeTrue, sizeof(attributeTrue) },         // Ключ предназначен для подписи
//	{ CKA_VERIFY, &attributeTrue, sizeof(attributeTrue) },       // Ключ предназначен для проверки подписи
// };

/*************************************************************************
 * Шаблон для поиска симметричного ключа ГОСТ 28147-89                    *
 *************************************************************************/
// CK_ATTRIBUTE secKeyTemplate[] =
//{
//	{ CKA_ID, &secretKeyId, sizeof(secretKeyId) - 1 },             // Идентификатор ключа
//	{ CKA_CLASS, &secretKeyObject, sizeof(secretKeyObject) },      // Класс - секретный ключ
//	{ CKA_KEY_TYPE, &keyTypeGost28147, sizeof(keyTypeGost28147) }, // Тип ключа - ГОСТ 28147-89
//	{ CKA_SIGN, &attributeTrue, sizeof(attributeTrue) },           // Ключ предназначен для подписи
//	{ CKA_VERIFY, &attributeTrue, sizeof(attributeTrue) },         // Ключ предназначен для проверки подписи
// };

int main(void) {
    HMODULE module;            // Хэндл загруженной библиотеки PKCS#11
    CK_SESSION_HANDLE session; // Хэндл открытой сессии

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID_PTR slots; // Массив идентификаторов слотов
    CK_ULONG slotCount;   // Количество идентификаторов слотов в массиве

    CK_MECHANISM_TYPE_PTR mechanisms; // Массив поддерживаемых механизмов
    CK_ULONG mechanismCount;          // Количество поддерживаемых механизмов

    CK_OBJECT_HANDLE_PTR signKeys; // Массив хэндлов ключей для выработки имитовставки
    CK_ULONG signKeysCount; // Количество хэндлов объектов в массиве

    CK_OBJECT_HANDLE_PTR verKeys; // Массив хэндлов ключей для проверки выработанной имитовставки
    CK_ULONG verKeysCount; // Количество хэндлов объектов в массиве

    CK_BYTE_PTR signature; // Указатель на буфер, содержащий имитовставку для данных
    CK_ULONG signatureSize; // Размер буфера, содержащего имитовставку для данных, в байтах

    CK_ULONG i; // Вспомогательная переменная-счетчик в циклах

    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11
    int r;    // Код возврата для функций, возвращающих int

    int isMacSupported = 0; // Флаг для проверки поддержки токеном механизма

    int errorCode = 1; // Флаг ошибки

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, exit);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Получить количество слотов c подключенными токенами                    *
     *************************************************************************/
    rv = functionList->C_GetSlotList(CK_TRUE, NULL_PTR, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList (number of slots)", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

    CHECK_AND_LOG(" Checking available tokens", slotCount > 0, " No tokens available", finalize_pkcs11);

    /*************************************************************************
     * Получить список слотов c подключенными токенами                        *
     *************************************************************************/
    slots = (CK_SLOT_ID_PTR)malloc(slotCount * sizeof(CK_SLOT_ID));
    CHECK(" Memory allocation for slots", slots != NULL, finalize_pkcs11);

    rv = functionList->C_GetSlotList(CK_TRUE, slots, &slotCount);
    CHECK_AND_LOG(" C_GetSlotList", rv == CKR_OK, rvToStr(rv), free_slots);
    printf(" Slots available: %d\n", (int)slotCount);

    /*************************************************************************
     * Получить список поддерживаемых токеном механизмов                      *
     *************************************************************************/
    rv = functionList->C_GetMechanismList(slots[0], NULL_PTR, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList (number of mechanisms)", rv == CKR_OK, rvToStr(rv), free_slots);

    CHECK_AND_LOG(" Checking mechanisms available", mechanismCount > 0, " No mechanisms available", free_slots);

    mechanisms = (CK_MECHANISM_TYPE_PTR)malloc(mechanismCount * sizeof(CK_MECHANISM_TYPE));
    CHECK(" Memory allocation for mechanisms", mechanisms != NULL_PTR, free_slots);

    rv = functionList->C_GetMechanismList(slots[0], mechanisms, &mechanismCount);
    CHECK_AND_LOG(" C_GetMechanismList", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Определение поддерживаемых токеном механизмов                          *
     *************************************************************************/
    for (i = 0; i < mechanismCount; ++i) {
        if (mechanisms[i] == CKM_KUZNECHIK_MAC) {
            isMacSupported = 1;
            break;
        }

        // if (mechanisms[i] == CKM_MAGMA_MAC) {
        //	isMacSupported = 1;
        //	break;
        // }

        // if (mechanisms[i] == CKM_GOST28147_MAC) {
        //	isMacSupported = 1;
        //	break;
        // }
    }

    CHECK_AND_LOG(" Checking CKM_KUZNECHIK_MAC support", isMacSupported, "CKM_KUZNECHIK_MAC isn`t supported!\n",
                  free_mechanisms);

    /*************************************************************************
     * Открыть сессию в первом доступном слоте                                *
     *************************************************************************/
    rv = functionList->C_OpenSession(slots[0], CKF_SERIAL_SESSION, NULL_PTR, NULL_PTR, &session);
    CHECK_AND_LOG(" C_OpenSession", rv == CKR_OK, rvToStr(rv), free_mechanisms);

    /*************************************************************************
     * Выполнить аутентификацию Пользователя                                  *
     *************************************************************************/
    rv = functionList->C_Login(session, CKU_USER, USER_PIN, USER_PIN_LEN);
    CHECK_AND_LOG(" C_Login", rv == CKR_OK, rvToStr(rv), close_session);
    printf("Initialization has been completed successfully.\n");

    /*************************************************************************
     * Сформировать имитовставку по алгоритму Кузнечик                        *
     *************************************************************************/
    printf("\nSigning data...\n");

    /*************************************************************************
     * Получить массив хэндлов секретных ключей                               *
     *************************************************************************/
    printf(" Getting MAC key...\n");
    r = findObjects(functionList, session, secKeyTemplate, arraysize(secKeyTemplate), &signKeys, &signKeysCount);
    CHECK(" findObjects", r == 0, logout);

    CHECK_AND_LOG(" Checking number of keys found", signKeysCount > 0, "No objects found\n", logout);

    /*************************************************************************
     * Инициализировать операцию выработки имитовставки Кузнечик              *
     *************************************************************************/
    rv = functionList->C_SignInit(session, &kuznechikMacMech, signKeys[0]);
    CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), free_signKeys);

    /*************************************************************************
     * Инициализировать операцию выработки имитовставки Магма                 *
     *************************************************************************/
    // rv = functionList->C_SignInit(session, &magmaMacMech, signKeys[0]);
    // CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), free_signKeys);

    /*************************************************************************
     * Инициализировать операцию выработки имитовставки ГОСТ 28147-89         *
     *************************************************************************/
    // rv = functionList->C_SignInit(session, &gost28147MacMech, signKeys[0]);
    // CHECK_AND_LOG("  C_SignInit", rv == CKR_OK, rvToStr(rv), free_signKeys);

    /*************************************************************************
     * Определить размер имитовставки                                         *
     *************************************************************************/
    rv = functionList->C_Sign(session, data, sizeof(data), NULL_PTR, &signatureSize);
    CHECK_AND_LOG("  C_Sign (get size)", rv == CKR_OK, rvToStr(rv), free_signKeys);

    /*************************************************************************
     * Выработать имитовставку                                                *
     *************************************************************************/
    signature = (CK_BYTE*)malloc(signatureSize * sizeof(CK_BYTE));
    CHECK("  Memory allocation for MAC", signature != NULL, free_signKeys);

    rv = functionList->C_Sign(session, data, sizeof(data), signature, &signatureSize);
    CHECK_AND_LOG("  C_Sign (signing)", rv == CKR_OK, rvToStr(rv), free_signature);

    /*************************************************************************
     * Распечатать буфер, содержащий имитовставку                             *
     *************************************************************************/
    printf("  MAC buffer is: \n");
    printHex(signature, signatureSize);
    printf("MAC has been computed successfully.\n");

    /*************************************************************************
     * Выполнить проверку выработанной имитовставки                           *
     *************************************************************************/
    printf("\nVerifying MAC...\n");

    /*************************************************************************
     * Получить массив хэндлов секретных ключей                               *
     *************************************************************************/
    printf(" Getting MAC key to verify...\n");
    r = findObjects(functionList, session, secKeyTemplate, arraysize(secKeyTemplate), &verKeys, &verKeysCount);
    CHECK(" findObjects", r == 0, logout);

    CHECK_AND_LOG(" Checking number of keys found", verKeysCount > 0, "No objects found\n", free_signature);

    /*************************************************************************
     * Инициализировать операцию проверки имитовставки Кузнечик               *
     *************************************************************************/
    rv = functionList->C_VerifyInit(session, &kuznechikMacMech, verKeys[0]);
    CHECK_AND_LOG(" C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_verKeys);

    /*************************************************************************
     * Инициализировать операцию проверки имитовставки Магма                  *
     *************************************************************************/
    // rv = functionList->C_VerifyInit(session, &magmaMacMech, verKeys[0]);
    // CHECK_AND_LOG(" C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_verKeys);

    /*************************************************************************
     * Инициализировать операцию проверки имитовставки ГОСТ 28147-89          *
     *************************************************************************/
    // rv = functionList->C_VerifyInit(session, &gost28147MacMech, verKeys[0]);
    // CHECK_AND_LOG(" C_VerifyInit", rv == CKR_OK, rvToStr(rv), free_verKeys);

    /*************************************************************************
     * Проверить имитовставку для данных                                      *
     *************************************************************************/
    rv = functionList->C_Verify(session, data, sizeof(data), signature, signatureSize);
    CHECK_AND_LOG("  C_Verify", rv == CKR_OK, rvToStr(rv), free_verKeys);

    printf("Verifying has been completed successfully.\n");

    /*************************************************************************
     * Выставить признак успешного завершения программы                       *
     *************************************************************************/
    errorCode = 0;

    /*************************************************************************
     * Выполнить действия для завершения работы с библиотекой PKCS#11         *
     *************************************************************************/
    printf("\nFinalizing... \n");

    /*************************************************************************
     * Очистить память, выделенную под вычисленную имитовставку,              *
     * объекты ключей                                                         *
     *************************************************************************/
free_verKeys:
    free(verKeys);

free_signature:
    free(signature);

free_signKeys:
    free(signKeys);

    /*************************************************************************
     * Сбросить права доступа                                                 *
     *************************************************************************/
logout:
    rv = functionList->C_Logout(session);
    CHECK_RELEASE_AND_LOG(" C_Logout", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Закрыть открытую сессию в слоте                                        *
     *************************************************************************/
close_session:
    rv = functionList->C_CloseSession(session);
    CHECK_RELEASE_AND_LOG(" C_CloseSession", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Очистить память, выделенную под механизмы                              *
     *************************************************************************/
free_mechanisms:
    free(mechanisms);

    /*************************************************************************
     * Очистить память, выделенную под слоты                                  *
     *************************************************************************/
free_slots:
    free(slots);

    /*************************************************************************
     * Деинициализировать библиотеку                                          *
     *************************************************************************/
finalize_pkcs11:
    rv = functionList->C_Finalize(NULL_PTR);
    CHECK_RELEASE_AND_LOG(" C_Finalize", rv == CKR_OK, rvToStr(rv), errorCode);

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
