/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*------------------------------------------------------------------------------------*
 * Пример работы с Рутокен при помощи библиотеки PKCS#11 на языке C                   *
 *------------------------------------------------------------------------------------*
 * Использование команд получения информации о событиях в слотах:                     *
 *  - инициализация библиотеки;                                                       *
 *  - проверка событий в слотах (без блокировки выполнения потока приложения);        *
 *  - ожидание события потоком в слотах (с блокировкой выполнения потока приложения). *
 *------------------------------------------------------------------------------------*
 * Данный пример является самодостаточным.                                            *
 *************************************************************************************/

#include <Common.h>

/*************************************************************************
 * Мьютекс для безопасного доступа к функции C_GetSlotInfo                *
 *************************************************************************/
static mutex_t getSlotInfoMutex;

/*************************************************************************
 * Запустить поток, ожидающий событие в слотах.                           *
 * До наступления события выполнение потока заблокировано                 *
 *************************************************************************/
void MonitoringSlots(void* param) // Указатель на структуру данных типа CK_FUNCTION_LIST_PTR
{
    CK_SLOT_ID slot;       // Идентификатор слота, в котором произошло событие
    CK_SLOT_INFO slotInfo; // Структура данных типа CK_SLOT_INFO с информацией о слоте

    int r;             // Код возврата для функций, возвращающих int
    CK_RV rv = CKR_OK; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    CK_FUNCTION_LIST_PTR functionList = param; // Указатель на список функций PKCS#11, хранящийся в структуре
                                               // CK_FUNCTION_LIST, получаемый из параметра функции

    for (;;) {
        /*************************************************************************
         * Ожидать событие в некотором слоте                                      *
         * (блокируя поток, НЕ используем флаг CKF_DONT_BLOCK)                    *
         *************************************************************************/
        slot = 0xFFFFFFFF;
        rv = functionList->C_WaitForSlotEvent(0, &slot, NULL_PTR);
        CHECK_AND_LOG(" C_WaitForSlotEvent (blocking)", (rv == CKR_OK) || (rv == CKR_CRYPTOKI_NOT_INITIALIZED),
                      rvToStr(rv), thread_exit);

        if (rv == CKR_CRYPTOKI_NOT_INITIALIZED) {
            printf(" PKCS#11 finalized.\n");
            break;
        }

        /*************************************************************************
         * Получить информацию о слоте                                            *
         *************************************************************************/
        r = lockMutex(&getSlotInfoMutex);
        ASSERT(" lockMutex", r == 0);

        rv = functionList->C_GetSlotInfo(slot, &slotInfo);

        r = unlockMutex(&getSlotInfoMutex);
        ASSERT(" unlockMutex", r == 0);

        if (rv == CKR_CRYPTOKI_NOT_INITIALIZED) {
            printf(" PKCS#11 finalized.\n");
            break;
        }
        CHECK_AND_LOG(" C_GetSlotInfo", rv == CKR_OK, rvToStr(rv), thread_exit);

        /*************************************************************************
         * Распечатать информацию о событии в слоте                               *
         *************************************************************************/
        printf("\n  Slot ID:          0x%8.8x \n", (int)slot);
        if (slotInfo.flags & CKF_TOKEN_PRESENT) {
            printf("  Token has been attached!\n");
        } else {
            printf("  Token has been detached!\n");
        }
    }

thread_exit:
    printf("Exiting from thread\n\n");
}

int main(void) {
    HMODULE module; // Хэндл загруженной библиотеки PKCS#11

    CK_FUNCTION_LIST_PTR functionList; // Указатель на список функций PKCS#11, хранящийся в структуре CK_FUNCTION_LIST
    CK_C_GetFunctionList getFunctionList; // Указатель на функцию C_GetFunctionList

    CK_SLOT_ID slot = 0xFFFFFFFF; // Идентификатор слота, в котором произошло событие
    CK_SLOT_INFO slotInfo; // Структура данных типа CK_SLOT_INFO с информацией о слоте

    int r;    // Код возврата для функций, возвращающих int
    CK_RV rv; // Код возврата. Могут быть возвращены только ошибки, определенные в PKCS#11

    int errorCode = 1; // Флаг ошибки

    uintptr_t thread = 0;  // Дескриптор потока
    int threadStarted = 0; // Флаг, сигнализирующий что поток запущен

    /*************************************************************************
     * Создать мьютекс                                                        *
     *************************************************************************/
    r = createMutex(&getSlotInfoMutex);
    CHECK(" createMutex", r == 0, exit);

    /*************************************************************************
     * Выполнить действия для начала работы с библиотекой PKCS#11             *
     *************************************************************************/
    printf("Initialization...\n");

    /*************************************************************************
     * Загрузить библиотеку                                                   *
     *************************************************************************/
    module = LoadLibrary(PKCS11ECP_LIBRARY_NAME);
    CHECK(" LoadLibrary", module != NULL, destroy_mutex);

    /*************************************************************************
     * Получить адрес функции запроса структуры с указателями на функции      *
     *************************************************************************/
    getFunctionList = (CK_C_GetFunctionList)GetProcAddress(module, "C_GetFunctionList");
    CHECK(" GetProcAddress", getFunctionList != NULL, unload_pkcs11);

    /*************************************************************************
     * Получить структуру с указателями на функции                            *
     *************************************************************************/
    rv = getFunctionList(&functionList);
    CHECK_AND_LOG(" Get function list", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    /*************************************************************************
     * Инициализировать библиотеку                                            *
     *************************************************************************/
    rv = functionList->C_Initialize(&initArgs);
    CHECK_AND_LOG(" C_Initialize", rv == CKR_OK, rvToStr(rv), unload_pkcs11);

    printf("\nPlease attach or detach Rutoken and press Enter...\n");
    (void)getchar();

    for (;;) {
        /*************************************************************************
         * Получить все события в слотах                                          *
         * (не блокируя поток, используем флаг CKF_DONT_BLOCK)                    *
         *************************************************************************/
        rv = functionList->C_WaitForSlotEvent(CKF_DONT_BLOCK, &slot, NULL_PTR);
        CHECK_AND_LOG(" C_WaitForSlotEvent (non blocking)", (rv == CKR_OK) || (rv == CKR_NO_EVENT), rvToStr(rv),
                      finalize_pkcs11);
        if (rv == CKR_NO_EVENT) {
            break;
        }

        /*************************************************************************
         * Получить информацию о слоте                                            *
         *************************************************************************/
        rv = functionList->C_GetSlotInfo(slot, &slotInfo);
        CHECK_AND_LOG(" C_GetSlotInfo", rv == CKR_OK, rvToStr(rv), finalize_pkcs11);

        printf(" Slot ID:           0x%8.8X \n", (int)slot);
        printf(" Slot description:  %.*s \n", (int)sizeof(slotInfo.slotDescription), slotInfo.slotDescription);
        printf(" Manufacturer:      %.*s \n", (int)sizeof(slotInfo.manufacturerID), slotInfo.manufacturerID);
        printf(" Flags:             0x%8.8X \n", (unsigned)slotInfo.flags);
        printf(" Hardware version:  %d.%d \n", slotInfo.hardwareVersion.major, slotInfo.hardwareVersion.minor);
        printf(" Firmware version:  %d.%d \n\n", slotInfo.firmwareVersion.major, slotInfo.firmwareVersion.minor);
    }

    /*************************************************************************
     * Запустить поток, ожидающий событие в каком-либо слоте.                 *
     * До наступления события выполнение запущенного потока заблокировано.    *
     * Первое же событие разблокирует выполнение ожидающего потока            *
     *************************************************************************/
    printf("\nStarting monitoring thread \n");

    r = createThread(&thread, &MonitoringSlots, functionList);
    CHECK_AND_LOG(" createThread", r == 0, "", finalize_pkcs11);
    threadStarted = 1;

    printf("\nPlease attach or detach Rutoken. Press Enter to exit.\n");

    (void)getchar();

    errorCode = 0;

    /*************************************************************************
     * Деинициализировать библиотеку и дождаться завершения выполнения потока *
     *************************************************************************/
finalize_pkcs11:
    r = lockMutex(&getSlotInfoMutex);
    ASSERT(" lockMutex", r == 0);

    rv = functionList->C_Finalize(NULL_PTR);
    ASSERT_AND_LOG_AND_ERRMSG(" C_Finalize", rv == CKR_OK, rvToStr(rv));

    r = unlockMutex(&getSlotInfoMutex);
    ASSERT(" unlockMutex", r == 0);

    if (threadStarted) {
        r = joinThread(thread);
        ASSERT_AND_LOG(" joinThread", r == 0);
    }

    /*************************************************************************
     * Выгрузить библиотеку из памяти                                         *
     *************************************************************************/
unload_pkcs11:
    CHECK_RELEASE(" FreeLibrary", FreeLibrary(module), errorCode);

    /*************************************************************************
     * Уничтожить мьютекс                                                     *
     *************************************************************************/
destroy_mutex:
    r = destroyMutex(&getSlotInfoMutex);
    CHECK_RELEASE(" destroyMutex", r == 0, errorCode);

exit:
    if (errorCode) {
        printf("\n\nSome error occurred. Sample failed.\n");
    } else {
        printf("\n\nSample has been completed successfully.\n");
    }

    return errorCode;
}
