/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

#ifndef RTFLASH_TYPES_H
#define RTFLASH_TYPES_H
/// \file rtflash-types.h
/// \brief Типы
#include "rtflash-macros.h"

#include <stddef.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#define RTFLASH_MIN_JOURNAL_SECTION_SIZE 500 ///< Минимальный размер журнального раздела (СБ/ФО) flash-памяти токена в мегабайтах

#define RTFLASH_LOCAL_PIN_MAX_SIZE 32      ///< Максимальный размер локального PIN-кода
#define RTFLASH_LOCAL_PIN_MIN_SIZE 6       ///< Минимальный размер локального PIN-кода
#define RTFLASH_PIN_MAX_SIZE 249           ///< Максимальный размер PIN-кода
#define RTFLASH_MAX_LOGIN_ATTEMPT_COUNT 10 ///< Максимальное количество неудачных попыток аутентификации
#define RTFLASH_MIN_LOGIN_ATTEMPT_COUNT 1  ///< Минимальное количество неудачных попыток аутентификации
#define RTFLASH_SECT_SEC_ATTR_COUNT 6      ///< Количество атрибутов безопасности раздела

#define RTFLASH_MVA_MAX_RECORD_SIZE 249  ///< Максимальный размер записи в МVА
#define RTFLASH_MVA_MAX_RECORD_COUNT 254 ///< Максимальное количество записей в МVА
#define RTFLASH_MVA_MAX_SIZE 33020       ///< Максимально возможный комбинированный размер МVА при его создании (количество записей * макс. размер записи)

typedef uint64_t rtflash_TokenSerial; ///< Серийный номер устройства
typedef uint8_t rtflash_RecordId;     ///< Идентификатор записи MVA
typedef uint8_t rtflash_SectionId;    ///< Идентификатор раздела
typedef uint32_t rtflash_MemSizeMB;   ///< Память в МБ
typedef uint64_t rtflash_UnixTime;    ///< Время в формате unix-time (неотрицательное целое число секунд с 1970.01.01.00:00:00)

/// Строка
typedef struct rtflash_String {
        size_t size; ///< Размер строки (без учета '\0').
        const char* ptr;
} rtflash_String;

typedef rtflash_String rtflash_PinCode;

/// Запись MVA - байтовая последовательность.
typedef struct rtflash_MvaRecord {
        size_t size;        ///< Размер последовательности.
        const uint8_t* ptr; ///< Указатель на последовательность.
} rtflash_MvaRecord;

/// Идентификатор секрета (#rtflash_Secret).
/// \attention Значение должно быть в диапазоне [#RTFLASH_SECRET_MIN_ID, #RTFLASH_SECRET_MAX_ID].
typedef uint16_t rtflash_SecretId;

/// Тип секрета (#rtflash_Secret).
typedef enum rtflash_SecretType {
        RTFLASH_SECRET_T_NONE = 0,      ///< Тип секрета не задан
        RTFLASH_SECRET_T_MVA = 1,       ///< MVA
        RTFLASH_SECRET_T_LOCAL_PIN = 2, ///< Локальный PIN-код
        RTFLASH_SECRET_T_UNKNOWN = 3    ///< Тип секрета не известен
} rtflash_SecretType;

/// Тип защиты от при создании локального PIN-кода.
typedef enum rtflash_SecureType {
        RTFLASH_SECURE_T_NONE = 0,     ///< Не заданно
        RTFLASH_SECURE_T_ADMIN = 1,    ///< PIN-код Администратора
        RTFLASH_SECURE_T_USER = 2,     ///< PIN-код Пользователя
        RTFLASH_SECURE_T_LOCAL_PIN = 3 ///< Локальный PIN-код
} rtflash_SecureType;

/// \brief Представляет собой объект для аутентификации.
/// Секрет - абстрактное понятие в рамках интерфейса библиотеки.
typedef struct rtflash_Secret {
        rtflash_SecretId id;
        rtflash_SecretType type;
} rtflash_Secret;

/// Секрет не задан
#define RTFLASH_SECRET_NONE \
        { 0, RTFLASH_SECRET_T_NONE }

#define RTFLASH_SECRET_MIN_ID 3     ///< Минимальный идентификатор секрета
#define RTFLASH_SECRET_MAX_ID 31    ///< Максимальный идентификатор секрета
#define RTFLASH_SECRET_MAX_COUNT 29 ///< Максимальное количество секретов

typedef rtflash_SecretId rtflash_MvaId;      ///< Идентификатор MVA
typedef rtflash_SecretId rtflash_LocalPinId; ///< Идентификатор локального PIN-кода

#define RTFLASH_Token_NULL RTFLASH_NULL_PTR ///< Невалидный объект rtflash_Token

typedef struct rtflash_TokenImpl_* rtflash_Token;

typedef enum rtflash_SectionType {
        RTFLASH_SECTION_USER = 0,           ///< Пользовательский раздел.
        RTFLASH_SECTION_JOURNAL_EVENTS = 1, ///< Журнал СБ.
        RTFLASH_SECTION_JOURNAL_ACCESS = 2, ///< Журнал ФО.
        RTFLASH_SECTION_ENCRYPTED = 3       ///< Зашифрованный раздел.
} rtflash_SectionType;

/// Режим доступа к разделу.
typedef enum rtflash_SectionAccessState {
        RTFLASH_SECTION_ACCESS_HI = 0, ///< Раздел скрыт.
        RTFLASH_SECTION_ACCESS_RO = 1, ///< Только чтение.
        RTFLASH_SECTION_ACCESS_RW = 3, ///< Полный доступ (чтение и запись).
        RTFLASH_SECTION_ACCESS_CD = 5  ///< CD-ROM.
} rtflash_SectionAccessState;

/// Глобальное условие защиты операции.
typedef enum rtflash_PrCondGlobal {
        RTFLASH_PR_COND_GLOBAL_FORBIDDEN = 0xFF,    ///< Операция не выполнима никогда.
        RTFLASH_PR_COND_GLOBAL_NONE = 0x00,         ///< Глобальное условие защиты отсутствует.
        RTFLASH_PR_COND_GLOBAL_ADMIN = 0x01,        ///< Аутентификация по PIN-коду Администратора
        RTFLASH_PR_COND_GLOBAL_USER = 0x02,         ///< Аутентификация по PIN-коду Пользователя
        RTFLASH_PR_COND_GLOBAL_ADMIN_OR_USER = 0x03 ///< Аутентификация по PIN-коду Администратора ИЛИ Пользователя
} rtflash_PrCondGlobal;

/// \brief Условия защиты операции.
/// Операцию можно защитить на глобальное условие защиты и/или аутентификацию по секрету.
/// Одновременно заданные глобальное условие и защита на секрет интерпретируются по условию "И".
/// \note Пример 1. { RTFLASH_PR_COND_GLOBAL_NONE, {} } => операция не защищена (доступна всем).
/// \note Пример 2. { RTFLASH_PR_COND_GLOBAL_FORBIDDEN, {} } => операция запрещена (не доступна никому).
/// \note Пример 3. { RTFLASH_PR_COND_GLOBAL_FORBIDDEN, { (secret.id > 0) } } => ошибка.
/// \note Пример 4. { RTFLASH_PR_COND_GLOBAL_ADMIN_OR_USER } => аутентификация по PIN-коду Администратора ИЛИ Пользователя.
/// \note Пример 5. { RTFLASH_PR_COND_GLOBAL_ADMIN_OR_USER, { 11, RTFLASH_SECRET_T_MVA } } => аутентификация по PIN-коду
/// Администратора ИЛИ Пользователя И аутентификация по указанному секрету (слепку ПК из MVA под идентификатором 11).
/// \note Пример 6. { RTFLASH_PR_COND_GLOBAL_NONE, { 11, RTFLASH_SECRET_T_MVA } } => аутентификация по указанному секрету
/// (слепку ПК из MVA под идентификатором 11).
typedef struct rtflash_PrConds {
        rtflash_PrCondGlobal global; ///< Глобальное условие защиты.
        rtflash_Secret secret;       ///< Секрет. Если secret.id = 0, то защита на секрет отсутствует.
} rtflash_PrConds;

/// Операция запрещена.
#define rtflash_PrConds_FORBIDDEN \
        RTFLASH_PR_COND_GLOBAL_FORBIDDEN, RTFLASH_SECRET_NONE

/// Операция не имеет условий защиты (доступна всем).
#define rtflash_PrConds_NONE \
        RTFLASH_PR_COND_GLOBAL_NONE, RTFLASH_SECRET_NONE

/// Операция защищена на глобальный PIN-код Администратора.
#define RTFLASH_PrConds_ADMIN_ONLY \
        RTFLASH_PR_COND_GLOBAL_ADMIN, RTFLASH_SECRET_NONE

/// Операция защищена на глобальный PIN-код Пользователя.
#define RTFLASH_PrConds_USER_ONLY \
        RTFLASH_PR_COND_GLOBAL_USER, RTFLASH_SECRET_NONE

/// Операция защищена на аутентификацию по секрету (#rtflash_Secret).
#define rtflash_PrConds_SECRET(secretId, secretType) \
        RTFLASH_PR_COND_GLOBAL_NONE, { secretId, secretType }

/// Операции смены режима доступа к разделу (защищенные операции).
typedef enum rtflash_SectionSecOp {
        RTFLASH_SECT_OP_HI_PERM = 0, ///< Скрыт, постоянно.
        RTFLASH_SECT_OP_RO_TEMP = 1, ///< Только чтение, временно.
        RTFLASH_SECT_OP_RO_PERM = 2, ///< Только чтение, постоянно.
        RTFLASH_SECT_OP_RW_TEMP = 3, ///< Чтение и запись, временно.
        RTFLASH_SECT_OP_RW_PERM = 4, ///< Чтение и запись, постоянно.
        RTFLASH_SECT_OP_CD_PERM = 5  ///< Режим CD-ROM, постоянно.
} rtflash_SectionSecOp;

/// Атрибуты безопасности раздела - список [операций смены режима доступа к разделу](#rtflash_SectionSecOp) с условиями
/// защиты перевода в конкретный режим. Для доступа к условиям защиты конкретной операции рекомендуется использовать
/// соответствующее значение rtflash_SectionSecOp.
/// \note Атрибут безопасности объекта файловой системы (ФС) Рутокен - совокупность защищенной операции и условий доступа к выполнению операции.
typedef rtflash_PrConds rtflash_SectionSecAttrs[RTFLASH_SECT_SEC_ATTR_COUNT];

/// Константа для установки атрибутов безопасности журнального раздела (все операции, кроме перевода в режим RO_TEMP, запрещены).
/// \note Для журнала RTFLASH_SECT_OP_RO_TEMP запрещено защищать на RTFLASH_PR_COND_GLOBAL_FORBIDDEN (нельзя запретить перевод в RO_TEMP).
#define rtflash_SectionSecAttrs_JOURNAL_DEFAULT      \
        /* HI_PERM */ { rtflash_PrConds_FORBIDDEN }, \
        /* RO_TEMP */ { rtflash_PrConds_NONE },      \
        /* RO_PERM */ { rtflash_PrConds_FORBIDDEN }, \
        /* RW_TEMP */ { rtflash_PrConds_FORBIDDEN }, \
        /* RW_PERM */ { rtflash_PrConds_FORBIDDEN }, \
        /* CD_PERM */ { rtflash_PrConds_FORBIDDEN }

/// Представляет раздел flash-памяти токена и информацию о разделе.
typedef struct rtflash_Section {
        rtflash_SectionId id;                    ///< Уникальный идентификатор раздела.
        rtflash_SectionType type;                ///< Тип раздела.
        rtflash_MemSizeMB size;                  ///< Размер раздела в МБ.
        rtflash_SectionAccessState accessRights; ///< Текущие права доступа к разделу.
        rtflash_SectionSecAttrs secAttrs;        ///< Атрибуты безопасности раздела.
} rtflash_Section;

/// \brief Представляет входные параметры для создания MVA.
/// \note Комбинированный размер (maxRecordCount * maxRecordSize) не должен превышать #RTFLASH_MVA_MAX_SIZE.
typedef struct rtflash_MvaParams {
        uint8_t maxRecordCount;  ///< Допустимый диапазон значений: [1, #RTFLASH_MVA_MAX_RECORD_COUNT].
        uint8_t maxRecordSize;   ///< Допустимый диапазон значений: [1, #RTFLASH_MVA_MAX_RECORD_SIZE].
        uint8_t maxAttemptCount; ///< Допустимый диапазон значений: [1, 10].
} rtflash_MvaParams;

/// Представляет сведения о ПК пользователя для записи в журнал СБ.
typedef struct rtflash_PcInfo {
        rtflash_String userName; ///< Имя пользователя ПК. Указатель на строку UTF-8, содержащую имя пользователя.
                                 // Если имя пользователя недоступно, необходимо передать RTFLASH_NULL_PTR.

        rtflash_String domainName; ///< Доменное имя ПК. Указатель на строку UTF-8, содержащую имя домена.
                                   // Если ПК не является частью домена, необходимо передать RTFLASH_NULL_PTR.

        rtflash_String osVersion; ///< Версия операционной системы ПК. Указатель на строку UTF-8, указывающую версию ОС.
                                  // Если версия ОС недоступна, необходимо передать RTFLASH_NULL_PTR.

        rtflash_String deviceName; ///< Имя ПК. Указатель на строку UTF-8, содержащую имя ПК.
                                   // Если имя ПК недоступно, необходимо передать RTFLASH_NULL_PTR.


        rtflash_UnixTime systemTime; ///< Системное время в формате unix-time.
} rtflash_PcInfo;

typedef uint16_t rtflash_Attempts;

#ifdef __cplusplus
}
#endif

#endif // RTFLASH_TYPES_H
