/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Смена атрибута доступа к разделу по PIN-коду Пользователя              *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - смена атрибута доступа к разделу по PIN-коду Пользователя.          *
 **************************************************************************
 * Требования:                                                            *
 *  - указать в константах значение текущего PIN-кода Пользователя;       *
 *  - запустить пример SplitIntoSections.c для формирования 4 разделов    *
 *    во flash-памяти токена.                                             *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kUserPinCode = { 8, "12345678" }; // Значение PIN-кода Пользователя

/**************************************************************************
 * Константы идентификаторов разделов, созданных при выполнении примера   *
 * SplitIntoSections.c                                                    *
 **************************************************************************/

static rtflash_SectionId kUserOnlyProtectedSectionId = 1;

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/
    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Смена атрибутов безопасности раздела с помощью PIN-кода Пользователя   *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Пользователя
    ST_OK_CHECK(rtflash_login_user(token, kUserPinCode), cleanup);

    // Смена атрибута безопасности раздела на чтение, постоянное
    ST_OK_CHECK(
        rtflash_change_section_access_rights(token, kUserOnlyProtectedSectionId, RTFLASH_SECTION_ACCESS_RO, true),
        cleanup);

    // Смена атрибута безопасности раздела на чтение, временное
    // При переподключении токена, атрибут безопасности будет установлен на тот, который был до смены
    ST_OK_CHECK(
        rtflash_change_section_access_rights(token, kUserOnlyProtectedSectionId, RTFLASH_SECTION_ACCESS_RO, false),
        cleanup);

    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);
    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожение объекта токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}