/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Смена атрибута доступа к разделу по PIN-коду Пользователя и записи     *
 * файла MVA                                                              *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - смена атрибута доступа к разделу с помощью PIN-кода Пользователя и  *
 *    записи в файле MVA.                                                 *
 **************************************************************************
 * Требования:                                                            *
 *  - указать в константах значение текущего PIN-кода Пользователя;       *
 *  - запустить пример SplitIntoSections.c для формирования 4 разделов    *
 *    во flash-памяти токена.                                             *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kUserPinCode = { 8, "12345678" }; // Значение PIN-кода Пользователя

/**************************************************************************
 * Константы идентификаторов разделов, созданных при выполнении примера   *
 * SplitIntoSections.c                                                    *
 **************************************************************************/
static rtflash_SectionId kUserWithMvaProtectedSectionId = 2;

int main() {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_MvaId mvaId = 0;                                // Идентификатор файла MVA
    rtflash_Section* tokenSectionsArray = RTFLASH_NULL_PTR; // Массив разделов flash-памяти на токене
    size_t tokenSectionsArrayLength = 0; // Размер массива разделов flash-памяти на токене
    rtflash_Section mvaProtectedSection; // Информация о разделе, защищенном с помощью записи в файле MVA

    rtfl_utils_PcSnapshot snapshot = { 0 }; // Слепок ПК
    rtflash_MvaRecord mvaRecord;            // Запись файла MVA

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /*************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                      *
     *************************************************************************/
    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Смена атрибутов безопасности раздела с помощью PIN-кода Пользователя и *
     * файла MVA                                                              *
     **************************************************************************
     * В примере SplitIntoSections.c был создан файл MVA и туда был добавлен  *
     * слепок ПК. Поэтому для текущего примера надо повторить вычисление      *
     * слепка и произвести по нему аутентификацию.                            *
     * Дополнительно идентификатор файла MVA будет находиться из информации о *
     * разделах flash-памяти                                                  *
     *************************************************************************/

    /**************************************************************************
     * Поиск идентификатора файла MVA                                         *
     **************************************************************************/

    // Получаем массив разделов flash-памяти
    ST_OK_CHECK(rtflash_create_section_list(token, &tokenSectionsArray, &tokenSectionsArrayLength), cleanup);

    // Получаем информацию о разделе, защищенном с помощью записи в файле MVA
    mvaProtectedSection = tokenSectionsArray[kUserWithMvaProtectedSectionId - 1];

    // Находим идентификатор файла MVA, на который был защищен раздел
    mvaId = mvaProtectedSection.secAttrs[RTFLASH_SECT_OP_RO_TEMP].secret.id;

    /**************************************************************************
     * Вычисление слепка ПК и добавление его в файл MVA                       *
     **************************************************************************/

    // Вычисляем слепок ПК
    UTILS_ST_OK_CHECK(rtfl_utils_pc_snapshot(&snapshot), cleanup);
    // Переводим слепок в запись файла MVA
    mvaRecord = (rtflash_MvaRecord){
        .size = snapshot.size,               // Размер записи
        .ptr = (const uint8_t*)snapshot.ptr, // Указатель на данные записи
    };

    // Аутентификация по слепку ПК
    ST_OK_CHECK(rtflash_login_mva(token, mvaId, mvaRecord), cleanup);
    // Аутентификация PIN-кодом Пользователя
    ST_OK_CHECK(rtflash_login_user(token, kUserPinCode), cleanup);

    /**************************************************************************
     * Смена атрибутов безопасности раздела                                   *
     **************************************************************************/

    // Смена атрибута безопасности раздела на чтение, постоянное
    ST_OK_CHECK(
        rtflash_change_section_access_rights(token, kUserWithMvaProtectedSectionId, RTFLASH_SECTION_ACCESS_RO, true),
        cleanup);

    // Смена атрибута безопасности раздела на чтение, временное
    // При переподключении токена, атрибут безопасности будет установлен на тот, который был до смены
    ST_OK_CHECK(
        rtflash_change_section_access_rights(token, kUserWithMvaProtectedSectionId, RTFLASH_SECTION_ACCESS_RO, false),
        cleanup);

    // Сбрасывание текущих прав
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Освобождаем память массива разделов flash-памяти
    if (tokenSectionsArray != RTFLASH_NULL_PTR)
        rtflash_destroy_section_list(&tokenSectionsArray);
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Очищаем память, выделенную под слепок ПК
    if (snapshot.ptr != RTFLASH_NULL_PTR) {
        rtfl_utils_StatusCode stCode = rtfl_utils_pc_snapshot_destroy(&snapshot);
        if (stCode != RTFL_UTILS_ST_OK) {
            printf("Cleanup error: rtfl_utils_pc_snapshot_destroy status code: %u\n", stCode);
            exitCode = EXIT_FAILURE;
        } else {
            snapshot = (rtfl_utils_PcSnapshot){ 0 };
        }
    }
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}