/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Очистка flash-памяти токена                                            *
 **************************************************************************
 * Библиотека rtflash не предоставляет средств для форматирования         *
 * flash-памяти токена. Данный пример будет демонстрирует создание        *
 * незащищенного пользовательского раздела во всей доступной памяти.      *
 * Такое переразбиение позволяет очистить flash-память от ранее созданных *
 * разделов.                                                              *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - формирование раздела, занимающего всю доступную flash-память;       *
 *  - разбиение на новый раздел.                                          *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значение текущего PIN-кода Администратора.     *
 **************************************************************************
 * После выполнения примера разделы подключенного токена будут изменены.  *
 * Журнальный раздел изменен не будет.                                    *
 *************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_MemSizeMB availableFlashMemory = 0; // Количество доступной для разбиения flash-памяти
    rtflash_Section sectionNoProtected; // Раздел, не имеющий защиты и занимающий всю доступную flash-память

    /*************************************************************************
     * Получение массива серийных номеров подключенных токенов               *
     *************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /*************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                      *
     *************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Получение доступного объема flash-памяти                               *
     *************************************************************************/

    // Получаем количество доступной для разбиения flash-памяти на токене
    ST_OK_CHECK(rtflash_get_available_flash_size(token, &availableFlashMemory), cleanup);

    // Выводим полученное значение на экран
    printf("Available flash size: %u MB\n", availableFlashMemory);

    /*************************************************************************
     * Формирование раздела, не имеющего защиты и занимающего всю доступную  *
     * память                                                                *
     *************************************************************************/

    sectionNoProtected = (rtflash_Section){
        .id = 0,                      // При создании раздела это поле игнорируется
        .type = RTFLASH_SECTION_USER, // Указываем, что раздел пользовательский
        .size = availableFlashMemory, // Размер раздела
        .accessRights = RTFLASH_SECTION_ACCESS_RW, // Текущие права доступа к разделу - чтение и запись
        .secAttrs = { // Все операции не имеют защиты
            { rtflash_PrConds_NONE },
            { rtflash_PrConds_NONE },
            { rtflash_PrConds_NONE },
            { rtflash_PrConds_NONE },
            { rtflash_PrConds_NONE },
            { rtflash_PrConds_NONE },
        },
    };

    /*************************************************************************
     * Замещение ранее созданных разделов разделом, не имеющим защиты        *
     *************************************************************************/

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    // Выполняем переразбиение на только что сформированные разделы
    ST_OK_CHECK(rtflash_split_into_sections(token, &sectionNoProtected, 1), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}