/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Создание журнала СБ (Событий Безопасности)                             *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - получение доступной для разбиения flash-памяти;                     *
 *  - формирование раздела журнала СБ;                                    *
 *  - создание раздела журнала СБ.                                        *
 **************************************************************************
 * Требования:                                                            *
 *  - указать в константах значение текущего PIN-кода Администратора;     *
 *  - на токене не должен существовать журнал СБ.                         *
 **************************************************************************
 * Выполнение примера приведет к созданию раздела журнала СБ.             *
 * Поскольку этот раздел является неудаляемым и существует в единственном *
 * экземпляре, то корректно этот пример может быть выполнен один раз.     *
 * Если вы не хотите создавать раздел журнала СБ не выполняйте            *
 * этот пример, ведь удалить такой раздел будет невозможно.               *
 * Все разделы, созданные ранее, будут удалены.                           *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы для выполнения этого примера                                 *
 **************************************************************************/
#define JOURNAL_SECTION_MIN_SIZE 500 // Минимальный размер раздела журнала СБ
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_MemSizeMB availableFlashMemory = 0; // Количество доступной для разбиения flash-памяти
    rtflash_Section journalEventsSection; // Раздел журнала СБ

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Получение доступного объема flash-памяти                               *
     **************************************************************************/

    // Получаем количество доступной для разбиения flash-памяти на токене
    ST_OK_CHECK(rtflash_get_available_flash_size(token, &availableFlashMemory), cleanup);

    // Выводим полученное значение на экран
    printf("Available flash size: %u\n", availableFlashMemory);

    /**************************************************************************
     * Формирование раздела журнала СБ                                        *
     **************************************************************************/

    journalEventsSection = (rtflash_Section){
        .id = 0, // При создании раздела это поле игнорируется
        .type = RTFLASH_SECTION_JOURNAL_EVENTS, // Указываем, что раздел - это журнал СБ
        .size = JOURNAL_SECTION_MIN_SIZE, // Для примера создадим журнальный раздел минимального размера
        .accessRights = RTFLASH_SECTION_ACCESS_HI, // Текущие права доступа - скрытый. С другими правами доступа журнал
                                                   // создать нельзя
        { rtflash_SectionSecAttrs_JOURNAL_DEFAULT }, // Ставим значение атрибутов раздела журнала СБ по умолчанию
    };

    /**************************************************************************
     * Создание раздела журнала СБ на токене                                  *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    // Выполняем переразбиение на только что сформированные разделы
    ST_OK_CHECK(rtflash_split_into_sections(token, &journalEventsSection, 1), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}