/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Создание и изменение файлов локального PIN-кода                        *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - аутентификация PIN-кодом Администратора;                            *
 *  - создание файла локального PIN-кода с защитой от изменения на        *
 *    PIN-код Администратора;                                             *
 *  - изменение значения созданного локального PIN-кода;                  *
 *  - создание файла локального PIN-кода с защитой от изменения на        *
 *    PIN-код Пользователя;                                               *
 *  - аутентификация PIN-кодом Пользователя;                              *
 *  - изменение значения созданного локального PIN-кода;                  *
 *  - создание файла локального PIN-кода с защитой от изменения на        *
 *    этот локальный PIN-код;                                             *
 *  - аутентификация созданным локальным PIN-кодом;                       *
 *  - изменение значения созданного локального PIN-кода.                  *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значение текущего PIN-кода Администратора.     *
 *  - токен должен быть хотя бы раз до выполнения примера отформатирован с*
 *    помощью библиотеки rtPKCS11ECP версии 2.18.1+ функцией              *
 *    C_EX_InitToken.                                                     *
 **************************************************************************
 * Данный пример является самодостаточным.                                *
 * В процессе выполнения кода данного примера на токене будут созданы     *
 * 3 файла локальных PIN-кодов.                                           *
 * Их удаление возможно только посредством форматирования токена.         *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора
static const rtflash_PinCode kUserPinCode = { 8, "12345678" }; // Значение PIN-кода Пользователя
static const rtflash_PinCode kLocalPin = { 8, "12345678" };    // Значение локального PIN-кода
static const rtflash_PinCode kNewLocalPin = { 8, "11111111" }; // Значение дополнительного локального PIN-кода

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_LocalPinId localPinAdminId;    // Идентификатор файла локального PIN-кода
    rtflash_LocalPinId localPinUserId;     // Идентификатор файла локального PIN-кода
    rtflash_LocalPinId localPinLocalPinId; // Идентификатор файла локального PIN-кода

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);

    /**************************************************************************
     * Создание файла локального PIN-кода с защитой от изменения              *
     * на PIN-код Администратора                                              *
     **************************************************************************/

    // Создаем файл локального PIN-кода
    ST_OK_CHECK(rtflash_create_local_pin(token,     // Токен, на котором создается файл
                                         kLocalPin, // Значение PIN-кода
                                         RTFLASH_SECURE_T_ADMIN, // Защита изменения на PIN-код Администратора
                                         &localPinAdminId),
                cleanup);

    printf("Local PIN file was created with id = %d\n", localPinAdminId);
    printf("\tThe PIN code value is protected from changes on the administrator key.\n");

    /**************************************************************************
     * Изменение файла локального PIN-кода с защитой от изменения             *
     * на PIN-код Администратора.                                             *
     * Дополнительная аутентификация не требуется, т.к. логин по PIN-коду     *
     * Администратора выполнен выше                                           *
     **************************************************************************/

    // Изменяем значение локального PIN-кода
    ST_OK_CHECK(rtflash_change_local_pin(token, // Токен на котором создан файл
                                         localPinAdminId, // Идентификатор файла локального PIN-кода
                                         kNewLocalPin), // Значение нового PIN-кода
                cleanup);

    printf("Local PIN file with id = %d was changed successfully\n", localPinAdminId);

    /**************************************************************************
     * Создание файла локального PIN-кода с защитой от изменения              *
     * на PIN-код Пользователя                                                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_local_pin(token,     // Токен, на котором создается файл
                                         kLocalPin, // Значение PIN-кода
                                         RTFLASH_SECURE_T_USER, // Защита изменения на PIN-код Пользователя
                                         &localPinUserId),
                cleanup);
    printf("\nLocal PIN file was created with id = %d\n", localPinUserId);
    printf("\tThe PIN code value is protected from changes on the user key.\n");

    /**************************************************************************
     * Изменение файла локального PIN-кода с защитой от изменения             *
     * на PIN-код Пользователя                                                *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Пользователя
    ST_OK_CHECK(rtflash_login_user(token, kUserPinCode), cleanup);
    // Выполняем изменение локального PIN-кода
    ST_OK_CHECK(rtflash_change_local_pin(token, // Токен на котором создан файл
                                         localPinUserId, // Идентификатор файла локального PIN-кода
                                         kNewLocalPin),
                cleanup); // Значение нового PIN-кода

    printf("Local PIN file with id = %d was changed successfully\n", localPinUserId);

    /**************************************************************************
     * Создание файла локального PIN-кода с защитой от изменения              *
     * на этот локальный PIN-код                                              *
     **************************************************************************/

    // Создаем локальный PIN-код
    ST_OK_CHECK(rtflash_create_local_pin(token,     // Токен, на котором создается файл
                                         kLocalPin, // Значение PIN-кода
                                         RTFLASH_SECURE_T_LOCAL_PIN, // Защита изменения на этот локальный PIN-код
                                         &localPinLocalPinId),
                cleanup);

    printf("\nLocal PIN file was created with id = %d\n", localPinLocalPinId);
    printf("\tThe PIN code value is protected from changes on itself.\n");

    /**************************************************************************
     * Изменение файла локального PIN-кода с защитой от изменения             *
     * на этот локальный PIN-код                                              *
     **************************************************************************/

    // Выполняем аутентификацию локальным PIN-кодом который необходимо изменить
    ST_OK_CHECK(rtflash_login_local_pin(token, localPinLocalPinId, kLocalPin), cleanup);
    // Выполняем изменение локального PIN-кода
    ST_OK_CHECK(rtflash_change_local_pin(token, // Токен, на котором создан файл
                                         localPinLocalPinId, // Идентификатор файла локального PIN-кода
                                         kNewLocalPin),
                cleanup); // Значение нового PIN-кода

    printf("Local PIN file with id = %d was changed successfully\n", localPinLocalPinId);

    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);
    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}