/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Создание и изменение файла MVA                                         *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - аутентификация PIN-кодом Администратора;                            *
 *  - создание пустого файла MVA;                                         *
 *  - добавление записи в файл MVA;                                       *
 *  - удаление записи из MVA;                                             *
 *  - удаление всех записей из MVA.                                       *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значение текущего PIN-кода Администратора.     *
 *  - токен должен быть хотя бы раз до выполнения примера отформатирован с*
 *    помощью библиотеки rtPKCS11ECP версии 2.18.1+ функцией              *
 *    C_EX_InitToken.                                                     *
 **************************************************************************
 * Данный пример является самодостаточным.                                *
 * В ходе выполнения кода примера на токене будет создан файл MVA.        *
 * Его удаление возможно только посредством форматирования.               *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    // Входные параметры для создания MVA
    rtflash_MvaParams params = {
        10, // Максимальное количество записей. Допустимый диапазон значений: [1, #RTFLASH_MVA_MAX_RECORD_COUNT]
        RTFLASH_MVA_MAX_RECORD_SIZE, // Максимальный размер записи. Допустимый диапазон значений: [1,
                                     // #RTFLASH_MVA_MAX_RECORD_SIZE]
        9, // Максимальное количество попыток аутентификации до блокировки. Допустимый диапазон значений: [1, 10]
    };

    rtflash_MvaId mvaId;               // Идентификатор файла MVA
    rtflash_RecordId recordId = 0;     // Идентификатор записи MVA
    rtflash_RecordId simpleRecordId;   // Идентификатор записи MVA
    rtflash_MvaRecord mvaRecord;       // Запись для добавления в MVA
    rtflash_MvaRecord mvaSimpleRecord; // Запись для добавления/удаления в MVA

    rtfl_utils_PcSnapshot snapshot = { 0 }; // Слепок ПК

    const uint8_t simpleRecord[] = {
        'R', 'u', 't', 'o', 'k', 'e', 'n', ' ', 'F', 'l', 'a', 's', 'h', '2'
    }; // Данные для записи

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);

    /**************************************************************************
     * Создание пустого файла MVA                                             *
     **************************************************************************/

    // Создание пустого файла MVA
    ST_OK_CHECK(rtflash_create_mva(token, &params, &mvaId), cleanup);
    printf("MVA file was created with id = %d\n", mvaId);

    /**************************************************************************
     * Добавление записи слепка ПК в файл MVA                                 *
     **************************************************************************/

    // Вычисляем слепок ПК
    UTILS_ST_OK_CHECK(rtfl_utils_pc_snapshot(&snapshot), cleanup);

    // Переводим слепок в запись файла MVA
    mvaRecord = (rtflash_MvaRecord){
        .size = snapshot.size,               // Размер записи
        .ptr = (const uint8_t*)snapshot.ptr, // Указатель на данные записи
    };

    // Записываем запись в файл MVA
    ST_OK_CHECK(rtflash_mva_add_record(token, mvaId, mvaRecord, &recordId), cleanup);

    /**************************************************************************
     * Удаление записи из MVA                                                 *
     **************************************************************************/

    mvaSimpleRecord = (rtflash_MvaRecord){ SIZE_OF(simpleRecord), simpleRecord };
    ST_OK_CHECK(rtflash_mva_add_record(token, mvaId, mvaSimpleRecord, &simpleRecordId),
                cleanup); // Добавление записи в MVA
    printf("Add record to MVA with id = %d. Record id = %d\n", mvaId, simpleRecordId);
    ST_OK_CHECK(rtflash_mva_delete_record(token, mvaId, simpleRecordId), cleanup); // Удаление записи из MVA
    printf("Delete record with id = %d from MVA with id = %d.\n", simpleRecordId, mvaId);

    /*************************************************************************:
     * Удаление всех записей из MVA                                           *
     **************************************************************************/

    ST_OK_CHECK(rtflash_mva_clear_all_records(token, mvaId), cleanup);
    printf("Delete all record from MVA with id = %d.\n", mvaId);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, выделенную под слепок ПК
    if (snapshot.ptr != RTFLASH_NULL_PTR) {
        rtfl_utils_StatusCode stCode = rtfl_utils_pc_snapshot_destroy(&snapshot);
        if (stCode != RTFL_UTILS_ST_OK) {
            printf("Cleanup error: rtfl_utils_pc_snapshot_destroy status code: %u\n", stCode);
            exitCode = EXIT_FAILURE;
        } else {
            snapshot = (rtfl_utils_PcSnapshot){ 0 };
        }
    }
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}