/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Получение развернутого описания ошибки                                 *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - получение кода ошибки;                                              *
 *  - вывод детальной информации об ошибке.                               *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значения, не соответствующие PIN-коду          *
 *    Пользователя.                                                       *
 **************************************************************************
 * При каждом выполнении примера количество оставшихся попыток            *
 * аутентификации уменьшается на 1.                                       *
 * PIN-код Пользователя может оказаться заблокированным.                  *
 **************************************************************************/

// Для корректного вывода int64_t на всех платформах
#include <inttypes.h>

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/

static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора
static const rtflash_PinCode kInvalidPinCode = { 2, "12" }; // Неверное значение PIN-кода Пользователя
static const rtflash_PinCode kExampleLocalPinCode = { 8, "12345678" }; // Значение локального PIN-кода

/**************************************************************************
 * Функция вывода на экран информации об ошибке вызова rtflash            *
 **************************************************************************/
static void printRtflashCallError(const char* call, const rtflash_StatusCode stCode) {
    const char* descr = RTFLASH_NULL_PTR;
    printf("Error calling %s\n\tStatus code: %u\n", call, stCode);
    if (rtflash_status_description(stCode, &descr) == RTFLASH_ST_OK && descr != RTFLASH_NULL_PTR)
        printf("\tDescription: %s\n", descr);
}

/**************************************************************************
 * Функция вывода на экран детальной информации об ошибке                 *
 **************************************************************************
 *  rtflash_StatusCode status - Код возврата последней функции rtflash    *
 **************************************************************************/
static int printDetailedDescription(const rtflash_StatusCode status) {
    int exitCode = EXIT_FAILURE;
    const char* statusCodeDescription = RTFLASH_NULL_PTR; // Описание кода возврата
    size_t conditionsLength = 0; // Размер массива дополнительной информации об ошибке
    rtflash_StatusCondition* conditionsArray = RTFLASH_NULL_PTR; // Массив дополнительной информации об ошибке
    size_t i = 0;
    const char* conditionDescription = RTFLASH_NULL_PTR; // Описание кода дополнительной информации об ошибке
    rtflash_StatusCondition condition;
    rtflash_ConditionDetail detail;

    // Получаем описание кода возврата
    {
        const rtflash_StatusCode stCode = rtflash_status_description(status, &statusCodeDescription);
        if (stCode != RTFLASH_ST_OK) {
            printRtflashCallError("rtflash_status_description", stCode);
            goto cleanup;
        }
    }
    // Выводим описание кода возврата
    printf("Status: %s\n", statusCodeDescription);

    // Получаем массив дополнительной информации об ошибке
    {
        const rtflash_StatusCode stCode = rtflash_last_status_conditions(&conditionsArray, &conditionsLength);
        if (stCode != RTFLASH_ST_OK) {
            printRtflashCallError("rtflash_last_status_conditions", stCode);
            goto cleanup;
        }
    }

    exitCode = EXIT_SUCCESS;
    // Проверяем, что массив дополнительной информации об ошибке не пуст
    if (conditionsLength == 0 || conditionsArray == RTFLASH_NULL_PTR)
        goto cleanup;

    // Начинаем вывод дополнительной информации об ошибке
    printf("Details:\n");
    for (i = 0; i < conditionsLength; ++i) {
        conditionDescription = RTFLASH_NULL_PTR;
        // Берем очередную дополнительную информацию об ошибке
        condition = conditionsArray[i];
        // Получаем представление дополнительной информации об ошибке
        detail = condition.detail;

        // Получаем описание кода дополнительной информации об ошибке
        {
            const rtflash_StatusCode stCode = rtflash_condition_description(condition.code, &conditionDescription);
            if (stCode != RTFLASH_ST_OK) {
                printRtflashCallError("rtflash_condition_description", stCode);
                exitCode = EXIT_FAILURE;
                goto cleanup;
            }
        }
        // Выводим номер дополнительной информации об ошибке, код и описание
        printf("\tCondition [%zu]: %s: ", i, conditionDescription);

        // Определяем тип хранящейся дополнительной информации об ошибке. Для безопасного доступа - в union
        if (detail.type == RTFLASH_DETAIL_char_ptr)
            // Выводим строковое представление дополнительной информации об ошибке
            printf("%s\n", detail.value.str ? detail.value.str : "(nil)");
        else
            // Выводим числовое представление дополнительной информации об ошибке
            printf("%" PRId64 "\n", detail.value.num);
    }

    // Удаляем сформированный массив дополнительной информации об ошибке
cleanup:
    if (conditionsArray != RTFLASH_NULL_PTR && conditionsLength > 0)
        rtflash_last_status_conditions_destroy(&conditionsArray, conditionsLength);
    return exitCode;
}

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_LocalPinId localPinId = 0;   // Идентификатор локального PIN-кода
    rtflash_Section sectionMvaProtected; // Раздел для генерации ошибки размера
    rtflash_StatusCode codeIncorrectUserLogin = RTFLASH_ST_OK;
    rtflash_StatusCode codeInvalidRights = RTFLASH_ST_OK;
    rtflash_StatusCode codeUnknownMvaID = RTFLASH_ST_OK;

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /***************************************************************************
     * Получение кода ошибки                                                   *
     ***************************************************************************
     * В этом блоке некоторые функции будут вызываться с неверными параметрами *
     * для намеренного получения кода ошибки и его последующей обработки.      *
     * Не повторяйте такие вызовы в реальном коде                              *
     ***************************************************************************/

    /***************************************************************************
     * Ошибка аутентификации по PIN-коду Пользователя                          *
     ***************************************************************************/

    // Используем неверный PIN-код для аутентификации
    codeIncorrectUserLogin = rtflash_login_user(token, kInvalidPinCode);
    // Выводим детализированную информацию об ошибке
    if (printDetailedDescription(codeIncorrectUserLogin) != EXIT_SUCCESS)
        goto cleanup;

    /***************************************************************************
     * Создание локального PIN-кода без прав пользователя                      *
     ***************************************************************************/

    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);
    // Создаем локальный PIN-код без прав пользователя
    codeInvalidRights = rtflash_create_local_pin(token, kExampleLocalPinCode, RTFLASH_SECURE_T_LOCAL_PIN, &localPinId);
    // Выводим детализированную информацию об ошибке
    if (printDetailedDescription(codeInvalidRights) != EXIT_SUCCESS)
        goto cleanup;

    /***************************************************************************
     * Размер раздела превышает объем flash-памяти                             *
     ***************************************************************************/

    sectionMvaProtected = (rtflash_Section){
        .id           = 0,                         // При создании раздела это поле игнорируется
        .type         = RTFLASH_SECTION_USER,      // Указываем, что раздел пользовательский
        .size         = 2395679283,                // Размер раздела, которого явно не существует
        .accessRights = RTFLASH_SECTION_ACCESS_RO, // Текущие права доступа к разделу - только чтение

        .secAttrs = {
            // Операция перевода в режим доступа: только чтение, временно
            // Защищена на PIN‑код Пользователя
            [RTFLASH_SECT_OP_RO_TEMP] = {
                .global = RTFLASH_PrConds_USER_ONLY,
            }
            // Все остальные операции - Не защищены
          },
      };

    // Выполняем аутентификацию PIN-кодом Администратора
    rtflash_login_admin(token, kAdminPinCode);
    // Пытаемся разбить на разделы
    codeUnknownMvaID = rtflash_split_into_sections(token, &sectionMvaProtected, 1);
    // Выводим детализированную информацию об ошибке
    if (printDetailedDescription(codeUnknownMvaID) != EXIT_SUCCESS)
        goto cleanup;
    // Сбрасываем текущие права
    rtflash_logout(token);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}