/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Получение информации о разделах во flash-памяти токена                 *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - получение списка разделов flash-памяти;                             *
 *  - получение информации о каждом разделе.                              *
 **************************************************************************
 * Данный пример является самодостаточным.                                *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Функция преобразования типа секрета rtflash в строку                   *
 **************************************************************************/
static const char* secretTypeToStr(rtflash_SecretType secureType) {
    switch (secureType) {
    case RTFLASH_SECRET_T_NONE:
        return "none";
    case RTFLASH_SECRET_T_MVA:
        return "mva";
    case RTFLASH_SECRET_T_LOCAL_PIN:
        return "local pin";
    default:
        return "unknown";
    }
}

/**************************************************************************
 * Функция преобразования типа раздела rtflash в строку                   *
 **************************************************************************/
static const char* sectionTypeToStr(rtflash_SectionType sectionType) {
    switch (sectionType) {
    case RTFLASH_SECTION_USER:
        return "User section";
    case RTFLASH_SECTION_ENCRYPTED:
        return "Encrypted section";
    case RTFLASH_SECTION_JOURNAL_ACCESS:
        return "Journal access section";
    case RTFLASH_SECTION_JOURNAL_EVENTS:
        return "Journal event section";
    default:
        return "Unknown section type";
    }
}

/**************************************************************************
 * Функция преобразования режима доступа к разделу flash-памяти в строку  *
 **************************************************************************/
static const char* accessStateToStr(rtflash_SectionAccessState accessState) {
    switch (accessState) {
    case RTFLASH_SECTION_ACCESS_HI:
        return "Hidden";
    case RTFLASH_SECTION_ACCESS_RO:
        return "ReadOnly";
    case RTFLASH_SECTION_ACCESS_RW:
        return "ReadWrite";
    case RTFLASH_SECTION_ACCESS_CD:
        return "CD-ROM";
    default:
        return "Unknown access state";
    }
}

/**************************************************************************
 * Функция преобразования глобального условия защиты раздела              *
 * flash-памяти в строку                                                  *
 **************************************************************************/
static const char* globPrCondsToStr(rtflash_PrCondGlobal accessState) {
    switch (accessState) {
    case RTFLASH_PR_COND_GLOBAL_FORBIDDEN:
        return "forbidden";
    case RTFLASH_PR_COND_GLOBAL_NONE:
        return "none";
    case RTFLASH_PR_COND_GLOBAL_ADMIN:
        return "admin";
    case RTFLASH_PR_COND_GLOBAL_USER:
        return "user";
    case RTFLASH_PR_COND_GLOBAL_ADMIN_OR_USER:
        return "admin or user";
    default:
        return "unknown";
    }
}

int main() {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_Section* tokenSectionsArray = RTFLASH_NULL_PTR; // Массив разделов flash-памяти на токене
    size_t tokenSectionsArrayLength = 0; // Размер массива разделов flash-памяти на токене
    rtflash_Section section;             // Текущий раздел

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Получение списка разделов flash-памяти токена                          *
     **************************************************************************/

    // Получаем массив разделов flash-памяти
    ST_OK_CHECK(rtflash_create_section_list(token, &tokenSectionsArray, &tokenSectionsArrayLength), cleanup);

    /**************************************************************************
     * Вывод информации о каждом разделе в консоль                            *
     **************************************************************************/

    // Перебираем все разделы
    for (i = 0; i < tokenSectionsArrayLength; ++i) {
        // Получаем раздел под номером i
        section = tokenSectionsArray[i];
        // Выводим идентификатор раздела
        printf("Section id: %hhu\n", section.id);
        // Выводим тип раздела
        printf("Section type: %s\n", sectionTypeToStr(section.type));
        // Выводим размер раздела
        printf("Section size: %u\n", section.size);
        // Выводим режим доступа к разделу
        printf("Section access state: %s\n", accessStateToStr(section.accessRights));
        // Начинаем вывод атрибутов безопасности раздела
        printf("Section attributes: \n");
        // Выводим требуемые права для перевода раздела в режим доступа: скрыт, постоянно
        printf("\tHidden permanent:     global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_HI_PERM].global),
               section.secAttrs[RTFLASH_SECT_OP_HI_PERM].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_HI_PERM].secret.type));

        // Выводим требуемые права для перевода раздела в режим доступа: только чтение, временно
        printf("\tRead only temporary:  global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_RO_TEMP].global),
               section.secAttrs[RTFLASH_SECT_OP_RO_TEMP].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_RO_TEMP].secret.type));

        // Выводим требуемые права для перевода раздела в режим доступа: только чтение, постоянно
        printf("\tRead only permanent:  global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_RO_PERM].global),
               section.secAttrs[RTFLASH_SECT_OP_RO_PERM].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_RO_PERM].secret.type));

        // Выводим требуемые права для перевода раздела в режим доступа: чтение и запись, временно
        printf("\tRead write temporary: global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_RW_TEMP].global),
               section.secAttrs[RTFLASH_SECT_OP_RW_TEMP].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_RW_TEMP].secret.type));

        // Выводим требуемые права для перевода раздела в режим доступа: чтение и запись, постоянно
        printf("\tRead write permanent: global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_RW_PERM].global),
               section.secAttrs[RTFLASH_SECT_OP_RW_PERM].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_RW_PERM].secret.type));

        // Выводим требуемые права для перевода раздела в режим доступа: CD-ROM, постоянно
        printf("\tCD-ROM permanent:     global: %s, secret id: %d, secret type: %s\n",
               globPrCondsToStr(section.secAttrs[RTFLASH_SECT_OP_CD_PERM].global),
               section.secAttrs[RTFLASH_SECT_OP_CD_PERM].secret.id,
               secretTypeToStr(section.secAttrs[RTFLASH_SECT_OP_CD_PERM].secret.type));
        printf("================================================================================\n");
    }

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Освобождаем память массива разделов flash-памяти
    if (tokenSectionsArray != RTFLASH_NULL_PTR)
        rtflash_destroy_section_list(&tokenSectionsArray);
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}
