/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Аутентификация на токене                                               *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - аутентификация PIN‑кодом Администратора;                            *
 *  - аутентификация PIN‑кодом Пользователя;                              *
 *  - создание файла локального PIN‑кода;                                 *
 *  - аутентификация локальным PIN‑кодом;                                 *
 *  - создание пустого файла MVA;                                         *
 *  - добавление записи в файл MVA;                                       *
 *  - аутентификация по записи в файле MVA;                               *
 *  - сброс всех прав;                                                    *
 *  - получение оставшегося количества попыток аутентификации             *
 *    для PIN‑кода Администратора;                                        *
 *  - получение оставшегося количества попыток аутентификации             *
 *    для PIN‑кода Пользователя;                                          *
 *  - получение оставшегося количества попыток аутентификации             *
 *    для локального PIN‑кода;                                            *
 *  - получение оставшегося количества попыток аутентификации для MVA;    *
 *  - аутентификация PIN‑кодом Администратора;                            *
 *  - сброс оставшегося количества попыток аутентификации для             *
 *    PIN‑кода Пользователя;                                              *
 *  - сброс оставшегося количества попыток аутентификации для             *
 *    локального PIN‑кода;                                                *
 *  - сброс оставшегося количества попыток аутентификации для MVA;        *
 *  - удаление объекта токена.                                            *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значения текущих PIN‑кодов Администратора и    *
 *    Пользователя.                                                       *
 *  - токен должен быть хотя бы раз до выполнения примера отформатирован с*
 *    помощью библиотеки rtPKCS11ECP версии 2.18.1+ функцией              *
 *    C_EX_InitToken.                                                     *
 **************************************************************************
 * Данный пример является самодостаточным.                                *
 * В процессе выполнения кода будут созданы файл локального PIN‑кода      *
 * и файл MVA. Их удаление возможно только посредством                    *
 * форматирования токена.                                                 *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN‑кода Администратора
static const rtflash_PinCode kUserPinCode = { 8, "12345678" }; // Значение PIN‑кода Пользователя
static const rtflash_PinCode kLocalPin = { 8, "12345678" };    // Значение локального PIN‑кода

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_LocalPinId localPinId = 0; // Идентификатор файла локального PIN‑кода

    rtflash_MvaParams params = {
        10,                          // Максимальное количество записей
        RTFLASH_MVA_MAX_RECORD_SIZE, // Максимальный размер записи
        9 // Максимальное количество попыток аутентификации до блокировки
    };
    rtflash_MvaId mvaId = 0;     // Идентификатор файла MVA
    rtflash_MvaId mvaIdSnap = 0; // Идентификатор файла MVA

    // Значение записи файла MVA
    const uint8_t record[] = { 'R', 'u', 't', 'o', 'k', 'e', 'n', ' ', 'F', 'l', 'a', 's', 'h' };
    rtflash_MvaRecord mvaRecord;       // Запись файла MVA
    rtflash_RecordId recordId = 0;     // Идентификатор записи
    rtflash_RecordId recordIdSnap = 0; // Идентификатор записи MVA

    rtfl_utils_PcSnapshot snapshot = { 0 }; // Слепок ПК
    rtflash_MvaRecord mvaRecordSnap;        // Запись MVA (слепок ПК)

    rtflash_Attempts attempts = 0;

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Аутентификация PIN‑кодом Администратора                                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    printf("Authentication with the Administrator PIN code was successful.\n");

    /**************************************************************************
     * Аутентификация PIN‑кодом Пользователя                                  *
     **************************************************************************/

    ST_OK_CHECK(rtflash_login_user(token, kUserPinCode), cleanup);
    printf("Authentication with the User PIN code was successful.\n");

    /**************************************************************************
     * Аутентификация локальным PIN‑кодом                                     *
     **************************************************************************/

    // Создаем файл локального PIN-кода
    ST_OK_CHECK(rtflash_create_local_pin(token,     // Токен
                                         kLocalPin, // Значение PIN‑кода
                                         RTFLASH_SECURE_T_USER, // Секрет, защищающий изменение локального PIN-кода
                                         &localPinId // Идентификатор сформированного локального PIN-кода
                                         ),
                cleanup);
    printf("\nLocal PIN file was created with id = %d\n", localPinId);

    // Аутентификация локальным PIN‑кодом
    ST_OK_CHECK(rtflash_login_local_pin(token, localPinId, kLocalPin), cleanup);
    printf("Authentication with the local PIN code was successful.\n");

    /**************************************************************************
     * Аутентификация по записи в файле MVA (вариант с произвольной записью)  *
     **************************************************************************/

    // Создаем файл MVA
    ST_OK_CHECK(rtflash_create_mva(token, &params, &mvaId), cleanup);
    printf("MVA file was created with id = %d\n", mvaId);

    mvaRecord = (rtflash_MvaRecord){ SIZE_OF(record), record };

    // Добавляем запись в файл MVA
    ST_OK_CHECK(rtflash_mva_add_record(token, mvaId, mvaRecord, &recordId), cleanup);
    printf("Add record to MVA with id = %d. Record id = %d\n", mvaId, recordId);

    // Аутентификация по записи в файле MVA
    ST_OK_CHECK(rtflash_login_mva(token, mvaId, mvaRecord), cleanup);
    printf("Authentication using MVA with random entry was successful.\n");

    /**************************************************************************
     * Аутентификация по записи в файле MVA (вариант со слепком ПК)           *
     **************************************************************************/

    // Создаем файл MVA
    ST_OK_CHECK(rtflash_create_mva(token, &params, &mvaIdSnap), cleanup);
    printf("MVA file was created with id = %d\n", mvaId);

    UTILS_ST_OK_CHECK(rtfl_utils_pc_snapshot(&snapshot), cleanup); // Вычисляем слепок ПК
    // Переводим слепок ПК в запись файла MVA
    mvaRecordSnap = (rtflash_MvaRecord){
        .size = snapshot.size,               // Размер записи
        .ptr = (const uint8_t*)snapshot.ptr, // Указатель на данные записи
    };

    // Добавляем запись в файл MVA
    ST_OK_CHECK(rtflash_mva_add_record(token, mvaIdSnap, mvaRecordSnap, &recordIdSnap), cleanup);
    printf("Add PC snapshot to MVA with id = %d. Record id = %d\n", mvaIdSnap, recordIdSnap);

    // Аутентификация по слепку ПК
    ST_OK_CHECK(rtflash_login_mva(token, mvaIdSnap, mvaRecordSnap), cleanup);
    printf("Authentication using MVA with PC snapshot was successful.\n");

    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    /**************************************************************************
     * Запрос оставшегося количества попыток аутентификации                   *
     * для разных объектов                                                    *
     **************************************************************************/

    // Для PIN‑кода Администратора
    ST_OK_CHECK(rtflash_get_remaining_login_attempts_admin(token, &attempts), cleanup);
    printf("Remaining attempts for Administrator PIN code: %d\n", attempts);

    // Для PIN‑кода Пользователя
    ST_OK_CHECK(rtflash_get_remaining_login_attempts_user(token, &attempts), cleanup);
    printf("Remaining attempts for User PIN code: %d\n", attempts);

    // Для локального PIN‑кода
    ST_OK_CHECK(rtflash_get_remaining_login_attempts_local_pin(token, localPinId, &attempts), cleanup);
    printf("Remaining attempts for local PIN (id = %d): %d\n", localPinId, attempts);

    // Для файла MVA
    ST_OK_CHECK(rtflash_get_remaining_login_attempts_mva(token, mvaId, &attempts), cleanup);
    printf("Remaining attempts for MVA (id = %d): %d\n", mvaId, attempts);

    /**************************************************************************
     * Сброс количества попыток аутентификации                                *
     **************************************************************************/

    // Аутентификация PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    printf("Authentication with the Administrator PIN code was successful.\n");

    // Сброс для PIN‑кода пользователя
    ST_OK_CHECK(rtflash_reset_login_attempts_user(token), cleanup);
    printf("Authentication attempts for User PIN were reset to the default value.\n");

    // Сброс для локального PIN‑кода
    ST_OK_CHECK(rtflash_reset_login_attempts_local_pin(token, localPinId), cleanup);
    printf("Authentication attempts for local PIN (id = %d) were reset to the default value.\n", localPinId);

    // Сброс для файла MVA
    ST_OK_CHECK(rtflash_reset_login_attempts_mva(token, mvaId), cleanup);
    printf("Authentication attempts for MVA (id = %d) were reset to the default value.\n", mvaId);

    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);
    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, выделенную под слепок ПК
    if (snapshot.ptr != RTFLASH_NULL_PTR) {
        rtfl_utils_StatusCode stCode = rtfl_utils_pc_snapshot_destroy(&snapshot);
        if (stCode != RTFL_UTILS_ST_OK) {
            printf("Cleanup error: rtfl_utils_pc_snapshot_destroy status code: %u\n", stCode);
            exitCode = EXIT_FAILURE;
        } else {
            snapshot = (rtfl_utils_PcSnapshot){ 0 };
        }
    }
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}