/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Разбиение flash-памяти токена на разделы                               *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - поиск первого доступного Рутокен Flash 3.0;                         *
 *  - установка соединения с Рутокен Flash 3.0;                           *
 *  - создание файла MVA на токене;                                       *
 *  - запись слепка ПК в файл MVA;                                        *
 *  - создание файла локального PIN-кода на токене;                       *
 *  - получение доступной для разбиения flash-памяти;                     *
 *  - формирование массива новых разделов;                                *
 *  - разбиение на новые разделы.                                         *
 **************************************************************************
 * Требование:                                                            *
 *  - указать в константах значения текущих PIN‑кодов Администратора и    *
 *    Пользователя.                                                       *
 *  - токен должен быть хотя бы раз до выполнения примера отформатирован с*
 *    помощью библиотеки rtPKCS11ECP версии 2.18.1+ функцией              *
 *    C_EX_InitToken.                                                     *
 **************************************************************************
 * После выполнения примера разделы на подключенном токене будут изменены.*
 * На токене будут созданы файл локального PIN-кода и файл MVA.           *
 **************************************************************************/

#include "common.h"

/**************************************************************************
 * Константы параметров файлов MVA для текущего примера                   *
 **************************************************************************/
#define EXAMPLE_MAX_ATTEMPTS_COUNT \
    RTFLASH_MAX_LOGIN_ATTEMPT_COUNT // Максимальное число неудачных попыток аутентификации используя MVA
#define EXAMPLE_MAX_MVA_RECORD_COUNT 10 // Максимальное число записей в файле MVA
#define EXAMPLE_MAX_MVA_RECORD_SIZE RTFLASH_MVA_MAX_RECORD_SIZE // Максимальный размер записи в файле MVA
#define SECTIONS_AMOUNT 4                                       // Количество новых разделов

/**************************************************************************
 * Константы различных PIN-кодов для этого примера                        *
 **************************************************************************/
static const rtflash_PinCode kLocalPinCode = { 8, "12345678" }; // Значение локального PIN-кода
static const rtflash_PinCode kAdminPinCode = { 8, "87654321" }; // Значение PIN-кода Администратора
static const rtflash_PinCode kUserPinCode = { 8, "12345678" }; // Значение PIN-кода Пользователя

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен
    bool isSupport = false;                 // Флаг, поддерживается ли токен
    size_t i = 0;                           // Счетчик

    rtflash_MvaId mvaId = 0; // Идентификатор файла MVA
    // Параметры для создания файла MVA
    rtflash_MvaParams mvaParams = { EXAMPLE_MAX_MVA_RECORD_COUNT, EXAMPLE_MAX_MVA_RECORD_SIZE,
                                    EXAMPLE_MAX_ATTEMPTS_COUNT };

    rtfl_utils_PcSnapshot snapshot = { 0 }; // Слепок ПК
    rtflash_MvaRecord mvaRecord;            // Запись MVA (слепок ПК)
    rtflash_RecordId recordId = 0;          // Идентификатор записи в файле MVA

    rtflash_LocalPinId localPinId = 0; // Идентификатор файла локального PIN-кода
    rtflash_MemSizeMB availableFlashMemory = 0; // Количество доступной для разбиения flash-памяти

    rtflash_Section sectionArray[SECTIONS_AMOUNT]; // Массив разделов
    rtflash_Section sectionNoSecretProtected;
    rtflash_Section sectionMvaProtected;
    rtflash_Section sectionLocalPinProtected;
    rtflash_Section sectionNoProtected;

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/

    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Поиск первого поддерживаемого библиотекой токена                       *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Если токен поддерживается, завершаем цикл
        if (isSupport)
            break;
        // Иначе уничтожаем объект с неподдерживаемым токеном и переходим к следующему
        rtflash_destroy_token(&token);
    }

    // Если нет поддерживаемых токенов
    if (!isSupport) {
        // Выводим сообщение: "Не найдено поддерживаемых токенов"
        printf("No supported tokens found.\n");
        goto cleanup;
    }

    /**************************************************************************
     * Получение идентификатора файла MVA на токене                           *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    // Иначе создаем новый файл MVA и получаем его идентификатор
    ST_OK_CHECK(rtflash_create_mva(token, &mvaParams, &mvaId), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    /**************************************************************************
     * Вычисление слепка ПК и добавление его в файл MVA                       *
     **************************************************************************/

    // Вычисляем слепок ПК
    UTILS_ST_OK_CHECK(rtfl_utils_pc_snapshot(&snapshot), cleanup);

    // Переводим слепок в запись файла MVA
    mvaRecord = (rtflash_MvaRecord){
        .size = snapshot.size,               // Размер записи
        .ptr = (const uint8_t*)snapshot.ptr, // Указатель на данные записи
    };

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    // Записываем запись в файл MVA
    ST_OK_CHECK(rtflash_mva_add_record(token, mvaId, mvaRecord, &recordId), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    /**************************************************************************
     * Получение идентификатора файла локального PIN-кода на токене           *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Пользователя
    ST_OK_CHECK(rtflash_login_user(token, kUserPinCode), cleanup);
    // Иначе создаем новый файл локального PIN-кода и получаем его идентификатор
    ST_OK_CHECK(rtflash_create_local_pin(token, kLocalPinCode, RTFLASH_SECURE_T_LOCAL_PIN, &localPinId), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    /**************************************************************************
     * Получение доступного объема flash-памяти                               *
     **************************************************************************/

    // Получаем количество доступной для разбиения flash-памяти на токене
    ST_OK_CHECK(rtflash_get_available_flash_size(token, &availableFlashMemory), cleanup);

    // Выводим полученное значение на экран
    printf("Available flash size: %u MB\n", availableFlashMemory);

    /**************************************************************************
     * Формирование новых разделов                                            *
     **************************************************************************
     * Будет сформировано 4 новых пользовательских раздела:                   *
     *  1. Раздел, не имеющий защиты операций на какой-то секрет (файл MVA    *
     *     или файл локального PIN-кода), но с защитой некоторых операций     *
     *     на PIN-код Пользователя.                                           *
     *  2. Раздел, у которого некоторые операции защищены на файл MVA, а      *
     *     некоторые на файл MVA + PIN-код Пользователя.                      *
     *  3. Раздел, у которого некоторые операции защищены на файл локального  *
     *     PIN-кода.                                                          *
     *  4. Раздел, не имеющий защиты ни на одну операцию.                     *
     * Размеры разделов будут 1/4 от доступной для разбиения flash-памяти     *
     **************************************************************************/

    // Раздел, не имеющий защиты на секрет
    sectionNoSecretProtected = (rtflash_Section){
        .id           = 0,                                            // При создании раздела это поле игнорируется
        .type         = RTFLASH_SECTION_USER,                         // Указываем, что раздел пользовательский
        .size         = availableFlashMemory / SECTIONS_AMOUNT,        // Размер раздела
        .accessRights = RTFLASH_SECTION_ACCESS_RW,                    // Текущие права доступа к разделу: чтение и запись

        .secAttrs = {
            // Операция перевода в режим доступа: скрыт, постоянно - Запрещена
            [RTFLASH_SECT_OP_HI_PERM] = { .global = rtflash_PrConds_FORBIDDEN },

            // Операция перевода в режим доступа: только чтение, временно - Защищена PIN‑кодом Пользователя
            [RTFLASH_SECT_OP_RO_TEMP] = { .global = RTFLASH_PrConds_USER_ONLY },

            // Операция перевода в режим доступа: только чтение, постоянно - Защищена PIN‑кодом Пользователя
            [RTFLASH_SECT_OP_RO_PERM] = { .global = RTFLASH_PrConds_USER_ONLY },

            // Операция перевода в режим доступа: чтение и запись, временно - Не защищена
            [RTFLASH_SECT_OP_RW_TEMP] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: чтение и запись, постоянно - Не защищена
            [RTFLASH_SECT_OP_RW_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: CD‑ROM, постоянно - Защищена PIN‑кодом Пользователя или
            // PIN‑код Администратора. Защита на секрет отсутствует
            [RTFLASH_SECT_OP_CD_PERM] = {
                .global = RTFLASH_PR_COND_GLOBAL_ADMIN_OR_USER,
                .secret = RTFLASH_SECRET_NONE,
            },
        },
    };

    // Раздел, защищенный записью в файле MVA
    sectionMvaProtected = (rtflash_Section){
        .id           = 0,                                            // При создании раздела это поле игнорируется
        .type         = RTFLASH_SECTION_USER,                         // Указываем, что раздел пользовательский
        .size         = availableFlashMemory / SECTIONS_AMOUNT,       // Размер раздела
        .accessRights = RTFLASH_SECTION_ACCESS_RO,                    // Текущие права доступа к разделу - только чтение

        .secAttrs = {
            // Операция перевода в режим доступа: скрыт, постоянно - Запрещена
            [RTFLASH_SECT_OP_HI_PERM] = { .global = rtflash_PrConds_FORBIDDEN },

            // Операция перевода в режим доступа: только чтение, временно - Защищена записью в файле MVA и PIN‑кодом Пользователя
            [RTFLASH_SECT_OP_RO_TEMP] = {
              .global = RTFLASH_PR_COND_GLOBAL_USER,
              .secret = { mvaId, RTFLASH_SECRET_T_MVA }, },

            // Операция перевода в режим доступа: только чтение, постоянно - Защищена записью в файле MVA и PIN‑кодом Пользователя
            [RTFLASH_SECT_OP_RO_PERM] = {
              .global = RTFLASH_PR_COND_GLOBAL_USER,
              .secret = { mvaId, RTFLASH_SECRET_T_MVA }, },

            // Операция перевода в режим доступа: чтение и запись, временно - Не защищена
            [RTFLASH_SECT_OP_RW_TEMP] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: чтение и запись, постоянно - Не защищена
            [RTFLASH_SECT_OP_RW_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: CD-ROM, постоянно - Запрещена
            [RTFLASH_SECT_OP_CD_PERM] = { .global = rtflash_PrConds_FORBIDDEN },
        },
    };

    // Раздел, защищенный локальным PIN-кодом
    sectionLocalPinProtected = (rtflash_Section){
        .id           = 0,                                            // При создании раздела это поле игнорируется
        .type         = RTFLASH_SECTION_USER,                         // Указываем, что раздел пользовательский
        .size         = availableFlashMemory / SECTIONS_AMOUNT,       // Размер раздела
        .accessRights = RTFLASH_SECTION_ACCESS_CD,                    // Текущие права доступа к разделу - CD-ROM

        .secAttrs = {
            // Операция перевода в режим доступа: скрыт, постоянно - Запрещена
            [RTFLASH_SECT_OP_HI_PERM] = {
                .global = rtflash_PrConds_FORBIDDEN,
            },

            // Операция перевода в режим доступа: только чтение, временно - Защищена локальным PIN‑кодом
            [RTFLASH_SECT_OP_RO_TEMP] = {
                .global = RTFLASH_PR_COND_GLOBAL_USER,
                .secret = { localPinId, RTFLASH_SECRET_T_LOCAL_PIN },
            },

            // Операция перевода в режим доступа: только чтение, постоянно - Защищена локальным PIN‑кодом
            [RTFLASH_SECT_OP_RO_PERM] = {
                .global = RTFLASH_PR_COND_GLOBAL_USER,
                .secret = { localPinId, RTFLASH_SECRET_T_LOCAL_PIN },
            },

            // Операция перевода в режим доступа: чтение и запись, временно - Не защищена
            [RTFLASH_SECT_OP_RW_TEMP] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: чтение и запись, постоянно - Не защищена
            [RTFLASH_SECT_OP_RW_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: CD-ROM, постоянно - Запрещена
            [RTFLASH_SECT_OP_CD_PERM] = { .global = rtflash_PrConds_FORBIDDEN },
        },
    };

    // Раздел не имеющий защиты
    sectionNoProtected = (rtflash_Section){
        .id           = 0,                                            // При создании раздела это поле игнорируется
        .type         = RTFLASH_SECTION_USER,                         // Указываем, что раздел пользовательский
        .size         = availableFlashMemory / SECTIONS_AMOUNT,       // Размер раздела
        .accessRights = RTFLASH_SECTION_ACCESS_HI,                    // Текущие права доступа к разделу - скрыт

        .secAttrs = {
            // Операция перевода в режим доступа: скрыт, постоянно - Не защищена
            [RTFLASH_SECT_OP_HI_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: только чтение, временно - Не защищена
            [RTFLASH_SECT_OP_RO_TEMP] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: только чтение, постоянно - Не защищена
            [RTFLASH_SECT_OP_RO_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: чтение и запись, временно - Не защищена
            [RTFLASH_SECT_OP_RW_TEMP] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: чтение и запись, постоянно - Не защищена
            [RTFLASH_SECT_OP_RW_PERM] = { .global = rtflash_PrConds_NONE },

            // Операция перевода в режим доступа: CD-ROM, постоянно - Не защищена
            [RTFLASH_SECT_OP_CD_PERM] = { .global = rtflash_PrConds_NONE },
        },
    };

    // Записываем сформированные разделы в массив разделов
    sectionArray[0] = sectionNoSecretProtected;
    sectionArray[1] = sectionMvaProtected;
    sectionArray[2] = sectionLocalPinProtected;
    sectionArray[3] = sectionNoProtected;

    /**************************************************************************
     * Переразбитие памяти токена на новые разделы                            *
     **************************************************************************/

    // Выполняем аутентификацию PIN-кодом Администратора
    ST_OK_CHECK(rtflash_login_admin(token, kAdminPinCode), cleanup);
    // Выполняем переразбиение на только что сформированные разделы
    ST_OK_CHECK(rtflash_split_into_sections(token, sectionArray, SECTIONS_AMOUNT), cleanup);
    // Сбрасываем текущие права
    ST_OK_CHECK(rtflash_logout(token), cleanup);

    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Очищаем память, выделенную под слепок ПК
    if (snapshot.ptr != RTFLASH_NULL_PTR) {
        rtfl_utils_StatusCode stCode = rtfl_utils_pc_snapshot_destroy(&snapshot);
        if (stCode != RTFL_UTILS_ST_OK) {
            printf("Cleanup error: rtfl_utils_pc_snapshot_destroy status code: %u\n", stCode);
            exitCode = EXIT_FAILURE;
        } else {
            snapshot = (rtfl_utils_PcSnapshot){ 0 };
        }
    }
    // Очищаем память, занятую массивом серийных номеров токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);
    // Уничтожаем объект токена перед завершением работы программы
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);
    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}