/*************************************************************************
* Rutoken                                                                *
* Copyright (c) 2003-2026, Aktiv-Soft JSC. All rights reserved.          *
* Подробная информация:  http://www.rutoken.ru                           *
*************************************************************************/

/**************************************************************************
 * Пример работы с Рутокен Flash 3.0 при помощи библиотеки rtflash на     *
 * языке C                                                                *
 **************************************************************************
 * Поиск поддерживаемых токенов                                           *
 **************************************************************************
 * Выполняемые действия:                                                  *
 *  - получение списка токенов в системе;                                 *
 *  - инициализация токенов;                                              *
 *  - проверка токенов на совместимость с текущей версией rtflash;        *
 *  - получение списка идентификаторов файлов MVA;                        *
 *  - получение списка идентификаторов файлов локальных PIN-кодов.        *
 **************************************************************************
 * Данный пример является самодостаточным.                                *
 **************************************************************************/

#include <inttypes.h>
#include "common.h"

int main(void) {
    int exitCode = EXIT_FAILURE;

    rtflash_TokenSerial* tokensSerialArray = RTFLASH_NULL_PTR; // Массив серийных номеров токенов
    size_t tokensSerialArrayLength = 0; // Размер массива серийных номеров токенов
    bool isTokensExists = false;        // Флаг, что список токенов не пуст

    bool isSupport = false; // Флаг, поддерживается ли токен
    size_t i = 0;           // Счетчик
    size_t j = 0;           // Счетчик

    rtflash_Token token = RTFLASH_NULL_PTR; // Объект токен

    rtflash_MvaId* mvaIdArray = RTFLASH_NULL_PTR; // Массив идентификаторов файлов MVA
    size_t mvaIdArrayLength = 0; // Размер массива идентификаторов файлов MVA

    rtflash_LocalPinId* localPinIdArray = RTFLASH_NULL_PTR; // Массив идентификаторов файлов локальных PIN-кодов
    size_t localPinIdArrayLength = 0; // Размер массива идентификаторов файлов локальных PIN-кодов

    /**************************************************************************
     * Получение массива серийных номеров подключенных токенов                *
     **************************************************************************/
    ST_OK_CHECK(rtflash_create_token_list(&tokensSerialArray, &tokensSerialArrayLength), cleanup);
    // Проверяем, что функция выполнилась успешно и массив не пуст
    isTokensExists = tokensSerialArray != RTFLASH_NULL_PTR && tokensSerialArrayLength > 0;
    EXPECT(isTokensExists, cleanup);

    /**************************************************************************
     * Вывод списка токенов с серийными номерами и статусом поддержки токена  *
     * библиотекой rtflash.                                                   *
     * Для каждого поддерживаемого токена выводится список идентификаторов,   *
     * записанных на него файлов локальных PIN-кодов и MVA                    *
     **************************************************************************/

    // Проходим по всему массиву токенов
    for (i = 0; i < tokensSerialArrayLength; ++i) {
        token = RTFLASH_NULL_PTR;
        mvaIdArray = RTFLASH_NULL_PTR;
        mvaIdArrayLength = 0;
        localPinIdArray = RTFLASH_NULL_PTR;
        localPinIdArrayLength = 0;

        // Создаем объект токена
        ST_OK_CHECK(rtflash_create_token(tokensSerialArray[i], &token), cleanup);
        // Проверяем, поддерживается ли этот токен
        ST_OK_CHECK(rtflash_is_token_supported(token, &isSupport), cleanup);
        // Выводим серийный номер токена и статус поддержки библиотекой
        printf("%zu) Token serial: %" PRIu64 ", supported: %s\n", i, (uint64_t)tokensSerialArray[i],
               (isSupport ? "true" : "false"));
        // Если токен не поддерживается, то переходим к следующему
        if (!isSupport) {
            // Уничтожаем объект токена
            rtflash_destroy_token(&token);
            continue;
        }

        /**************************************************************************
         * Получение списка идентификаторов файлов MVA,                           *
         * присутствующих на токене в данный момент                               *
         **************************************************************************/

        // Получаем массив идентификаторов файлов MVA
        ST_OK_CHECK(rtflash_create_mva_id_list(token, &mvaIdArray, &mvaIdArrayLength), cleanup);

        // Проверяем, что массив идентификаторов файлов MVA не пуст
        if (mvaIdArray != RTFLASH_NULL_PTR && mvaIdArrayLength > 0) {
            printf("\tMVA id: ");

            // Проходим по всем элементам массива
            for (j = 0; j < mvaIdArrayLength; ++j) {
                // Выводим идентификатор файла MVA
                printf("%d ", mvaIdArray[j]);
            }

            printf("\n");
        }
        /**************************************************************************
         * Получение списка идентификаторов файлов локальных PIN-кодов,           *
         * присутствующих на токене в данный момент                               *
         **************************************************************************/

        // Создаем массив идентификаторов файлов локальных PIN-кодов
        ST_OK_CHECK(rtflash_create_local_pin_id_list(token, &localPinIdArray, &localPinIdArrayLength), cleanup);

        // Проверяем, что массив идентификаторов файлов локальных PIN-кодов не пуст
        if (localPinIdArray != RTFLASH_NULL_PTR && localPinIdArrayLength > 0) {
            printf("\tLocal PIN id: ");

            // Проходим по всем элементам массива
            for (j = 0; j < localPinIdArrayLength; ++j) {
                // Выводим идентификатор файла локального PIN-кода
                printf("%d ", localPinIdArray[j]);
            }

            printf("\n");
        }
        // Очищаем память, занимаемую массивом идентификаторов файлов MVA
        if (mvaIdArray != RTFLASH_NULL_PTR)
            rtflash_destroy_mva_id_list(&mvaIdArray);
        // Очищаем память, занимаемую массивом идентификаторов файлов локальных PIN-кодов
        if (localPinIdArray != RTFLASH_NULL_PTR)
            rtflash_destroy_local_pin_id_list(&localPinIdArray);
        // Уничтожаем объект токена
        if (token != RTFLASH_NULL_PTR)
            rtflash_destroy_token(&token);
    }
    // Успешно завершаем работу программы
    exitCode = EXIT_SUCCESS;

cleanup:
    // Если произошла ошибка внутри цикла, то освобождаем остатки ресурсов текущей итерации
    if (mvaIdArray != RTFLASH_NULL_PTR)
        rtflash_destroy_mva_id_list(&mvaIdArray);
    if (localPinIdArray != RTFLASH_NULL_PTR)
        rtflash_destroy_local_pin_id_list(&localPinIdArray);
    if (token != RTFLASH_NULL_PTR)
        rtflash_destroy_token(&token);

    // Очищаем память, занимаемую массивом серийных номеров подключенных токенов
    if (tokensSerialArray != RTFLASH_NULL_PTR)
        rtflash_destroy_token_list(&tokensSerialArray);

    printf("Cleanup finished.\n");
    if (exitCode == EXIT_SUCCESS)
        printf("Sample completed successfully!\n");
    return exitCode;
}