#!/usr/bin/env bash

set -eu

# Для запуска скрипта необходимо выставить следующие параметры:
#vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# Путь до дистрибутива продукта (deb/rpm пакет rutokenlogon)
RUTOKENLOGON_PACKAGE='path/to/rutokenlogon_1.0.0-astra1_arm64.deb'
# Путь до deb/rpm пакета Guardant Control Center
GRDCONTROL_PACKAGE=''
# Путь, по которому будет сохранен пользовательский дистрибутив
USER_DISTRIBUTIVE_OUPUT_DIR="$(pwd)"
# Тип домена. Возможные значения: ipa, samba, aldpro и ad
DOMAIN_TYPE=""
# Путь до корневого сертификата или цепочки корневых сертификатов
PATH_TO_CA=""
# Адрес OCSP сервера
OCSP_URL=""
# Адрес по которому можно выгрузить CRL
CRL_URL=""
# Адрес сервера лицензирования
LICENSE_SERVER_URL=""
#^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

echo "Checking the environment and params..."

[[ $# -ne 0 ]] && echo "No arguments required" && exit 1

[[ -z "$RUTOKENLOGON_PACKAGE" ]] &&
echo "RUTOKENLOGON_PACKAGE param must be set" && exit 1
[[ -z "$DOMAIN_TYPE" ]] &&
echo "DOMAIN_TYPE param must be set" && exit 1

[[ "$DOMAIN_TYPE" != "ipa" ]] &&
[[ "$DOMAIN_TYPE" != "samba" ]] &&
[[ "$DOMAIN_TYPE" != "aldpro" ]] &&
[[ "$DOMAIN_TYPE" != "ad" ]] &&
echo "Invalid value of the DOMAIN_TYPE param: $DOMAIN_TYPE" && exit 1

[[ -z "$PATH_TO_CA" ]] &&
[[ "$DOMAIN_TYPE" != "ipa" ]] &&
[[ "$DOMAIN_TYPE" != "aldpro" ]] &&
echo "Path to CA certificate must be specified for this domain type: $DOMAIN_TYPE" && exit 1

[[ -n "$PATH_TO_CA" ]] &&
echo "Checking the CA certificate" &&
! openssl x509 -in "$PATH_TO_CA" -inform PEM > /dev/null &&
echo "CA path is not valid" && exit 1

[[ ! -f "$RUTOKENLOGON_PACKAGE" ]] && echo "Invalid rutokenlogon package path" && exit 1

[[ "$RUTOKENLOGON_PACKAGE" != *.deb ]] &&
[[ "$RUTOKENLOGON_PACKAGE" != *.rpm ]] &&
echo "Invalid rutokenlogon package" && exit 1

if [[ "$GRDCONTROL_PACKAGE" ]]; then
    [[ ! -f "$GRDCONTROL_PACKAGE" ]] && echo "Invalid grdcontrol package path" && exit 1

    [[ "$GRDCONTROL_PACKAGE" != *.deb ]] &&
    [[ "$GRDCONTROL_PACKAGE" != *.rpm ]] &&
    echo "Invalid grdcontrol package format" && exit 1

    ! [[ "$LICENSE_SERVER_URL" ]] && echo "LICENSE_SERVER_URL is required" && exit 1

    echo "Checking LICENSE_SERVER_URL" &&
    ! ping -c1 -t5 "$LICENSE_SERVER_URL" > /dev/null &&
    echo "Cannot reach license server" && exit 1
else
    [[ "$LICENSE_SERVER_URL" ]] && echo "LICENSE_SERVER_URL should be used only with GRDCONTROL_PACKAGE" && exit 1
fi


[[ "$CRL_URL" ]] &&
echo "Checking CRL_URL" &&
! ( wget -q -O - "$CRL_URL" | openssl crl -text -inform DER ) > /dev/null &&
echo "Can't download crl from CRL_URL" && exit 1

[[ "$OCSP_URL" ]] &&
echo "Checking OCSP_URL" &&
! wget -q -O - "$OCSP_URL" &&
echo "Cannot reach ocsp server" && exit 1

echo "Creating an temporary folder"
WORKDIR="$(mktemp -d)"
trap "rm -rf ${WORKDIR}" EXIT

OUTPUT_DIR="$WORKDIR/rutokenlogon_distributive"
mkdir -p "$OUTPUT_DIR"

[[ ! -z "$PATH_TO_CA" ]] &&
echo "Copying the CA certificate" &&
CA_NAME=$(basename "$PATH_TO_CA") &&
cp "$PATH_TO_CA" "${OUTPUT_DIR}/${CA_NAME}"

echo "Creating a user script"
USER_SCRIPT_PATH="${OUTPUT_DIR}/install_rutokenlogon.sh"
cat > "${USER_SCRIPT_PATH}" << EOF
#!/usr/bin/env bash
set -eu

SCRIPT_DIR=\$(dirname \$BASH_SOURCE)
RUTOKENLOGON_PACKAGE="\$SCRIPT_DIR/$(basename "$RUTOKENLOGON_PACKAGE")"
SSSD_CONF="/etc/sssd/sssd.conf"

[[ "\$(whoami)" != "root" ]] && echo "Run script as root" && exit 1

! [[ -f "\$SSSD_CONF" ]] && echo "Sssd configuration not found" && exit 1

echo "Install packages"
if [[ "\$RUTOKENLOGON_PACKAGE" == *.deb ]]; then
    apt-get update
    apt-get install -y krb5-pkinit "\$SCRIPT_DIR/"*.deb
elif [[ "\$RUTOKENLOGON_PACKAGE" == *alt* ]]; then
    apt-get update
    apt-get install -y "\$SCRIPT_DIR/"*.rpm
else
    dnf install --refresh -y krb5-pkinit "\$SCRIPT_DIR/"*.rpm
fi

function append_cert_ver_opt() {
    local opt=\$1

    if grep -qE '^\\s*certificate_verification\\s*=\s*\\w' "\$SSSD_CONF"; then
        sed -i "s~^\\s*certificate_verification\\s*=.*~\\0, \$opt~g" "\$SSSD_CONF"
    else
        echo -e "\\n[sssd]\\ncertificate_verification=\$opt" >> "\$SSSD_CONF"
    fi
}

function remove_cert_ver_opt() {
    local opt=\$1

    sed -i "s~^\\(\\s*certificate_verification\\s*=.*\\)\$opt\\s*,~\\1~g" "\$SSSD_CONF"
    sed -i "s~^\\(\\s*certificate_verification\\s*=.*\\),\\?\\s*\$opt~\\1~g" "\$SSSD_CONF"
}

echo "Remove no_verification option"
remove_cert_ver_opt "no_verification"

if [[ "$CRL_URL" ]]; then
    CRL_PATH=/etc/ipa/crl.pem

    echo "Download CRL"
    ! (wget -q -O - "$CRL_URL" | openssl crl -inform DER -out "\$CRL_PATH") && echo "Can't retrieve CRL" && exit 1

    echo "Add CRL autoupdate cron task"
    CRON_JOB="0 */3 * * *  wget -O - '$CRL_URL' | openssl crl -inform DER -out '\$CRL_PATH'"
    (crontab -l 2>/dev/null; echo "\$CRON_JOB") | sort -u | crontab -

    remove_cert_ver_opt "crl_file\\s*=[^,]*"

    echo "Setup SSSD for CRL"
    append_cert_ver_opt "crl_file=\$CRL_PATH"
fi

if [[ "$OCSP_URL" ]]; then
    echo "Checking OCSP" &&
    ! wget -q -O - "$OCSP_URL" &&
    echo "Can't get connect to OCSP" && exit 1

    remove_cert_ver_opt "no_ocsp"
    remove_cert_ver_opt "ocsp_default_responder\\s*=[^,]*"

    echo "Setup SSSD for custom OCSP"
    append_cert_ver_opt "ocsp_default_responder=$OCSP_URL"
fi

echo "Configure rtlogon"
rtlogon-cli configure --domain "${DOMAIN_TYPE}" \
 ${CA_NAME:+--ca-cert "\$SCRIPT_DIR/$CA_NAME"} \
 ${LICENSE_SERVER_URL:+--license-server "$LICENSE_SERVER_URL"}
EOF

chmod +x "${USER_SCRIPT_PATH}"

echo "Copying found rutokenlogon package"
cp "$RUTOKENLOGON_PACKAGE" "$OUTPUT_DIR"

[[ "$GRDCONTROL_PACKAGE" ]] &&
echo "Copying found grdcontrol package" &&
cp "$GRDCONTROL_PACKAGE" "$OUTPUT_DIR"

OUTPUT_ARCHIVE="$USER_DISTRIBUTIVE_OUPUT_DIR/rutokenlogon_distributive_deployment.tar"

pushd "$WORKDIR" > /dev/null
echo "Compressing a temporary folder into an archive"
tar -cf "$OUTPUT_ARCHIVE" *
popd > /dev/null

echo "Rtlogon package sucessfully created: $OUTPUT_ARCHIVE"
